CREATE OR REPLACE PACKAGE Sim2_Import_Result_Pkg IS
/* Updated 2/13/02        by SF
   
   
*/   
  -- Author  : KHENAS
  -- Created : 11/15/00 11:35:05 AM
  -- Purpose : import samples and results data into SIM tables from a text file

  /* KMS

  Package that imports activity and result records into SIM2 from a text file.
  Validates items within the text file to ensure that they are STORET friendly -
  otherwise an error message is issued and the data will be prevented from being
  exported to STORET without further manipulation (either through the SIM2 interface
  or by reloading).

  Before the package is run, the form (IMPF2_D) reads the text file into the
  SIM_IMP_LINES table so that each line is now stored as a separate record.  In
  addition, the import log for the import is created in the form.

  The goal in coding this package was to make it as modularized as possible.  Thus,
  most of the work is done in separate functions and procedures.  However, more than
  half of the package consists of the main import procedure.  This procedure gets basic,
  core information needed to load the data into SIM2 and to run the rest of the
  functions and procedures in the package.  Such information includes the is_number for
  the organization that is running the import, the delimiter used in the text file, the
  number of columns included in the import, the defaults for these columns, and the actual
  positions of the columns within the text file.  The actual positions of the columns are
  stored in table indexed by the default position of that columns.  Thus, for results,
  this table contains 57 entries which are either null (in the case of columns that have
  not been included in the import) or the actual position of the column in the line.

  Once this core information has been retrieved, the procedure begins a loop through the
  lines that have been imported into SIM_IMP_LINES.  Line specific variables are then
  adjusted (i.e. incremented in the case of line number), and the line checked to insure
  that it contains the correct number of items (columns).  The line is then parsed into
  separate items to be loaded, these items are stored in v_line_table, which is indexed
  by the actual position of the item within the text line.

  The procedure then loops through the columns (items) in the line.  It sets up variables
  that will be used later to determine what the item is and what should be done with it.
  This process is performed in a large if..then statement.  All blocks of this code are
  only entered if the item name is correct for that particular block.  While it is easy to
  forget that this part of the code is an if..then statement and to think instead that all
  lines of code are accessed in the order they appear in the procedure, it is important to
  remember that this code is processed is according to the processing order worked out in
  the sim_imp_cols table.

  Each block of code in the if..then statement deals with a particular item/column.  Within
  these blocks, the data that was brought in from the text file is prepared for loading into
  SIM2 and check against the requirements of STORET.  In many cases, this involves validating
  the item against the SIM2 allowable values table (created for items whose values have been
  limited by the STORET interface rather than a STORET table) or validated against values that
  are stored in STORET tables.  In both of these validation procedures, the translation table
  is checked before the items are validated.  In addition to these validations, items are
  validated against the data that has already been loaded from the line being imported.  This
  includes checks to insure that the activity being imported occured between the trip start
  and end dates as well as similar integrity concerns.

  Following the if..then statement, any other integrity checks/validations that have not yet
  been taken care of are performed.  These includes making sure that results are valid for the
  characteristic chosen for it.

  The activities and results are created by assigning the values that were obtained earlier in
  the program to records containing items for every field in the SIM tables for activities and
  results.  These records are then passed to functions that insert the data into the tables.

  Once the procedure has looped through all of the lines, it counts the number that were approved
  for export to STORET and updates all tables that are related to this.

  KMS */

   v_char_result_type               VARCHAR2(4);  -- dcl need this global for char type tracking

   TYPE col_table_type IS TABLE OF VARCHAR2(256) INDEX BY BINARY_INTEGER;

   TYPE col_table_num_type IS TABLE OF NUMBER(8) INDEX BY BINARY_INTEGER;

   TYPE t_activity_count IS TABLE OF NUMBER(6) INDEX BY BINARY_INTEGER;

   TYPE t_storet_pk_type IS RECORD (
     is_number         NUMBER(8),
     org_id            VARCHAR2(8)
     );

   TYPE t_acode_defs_rec IS RECORD (
     char_pk             t_storet_pk_type,
     anlpr_pk            t_storet_pk_type,
     labprep_pk          t_storet_pk_type,
     uom_pk              t_storet_pk_type,
     lab_pk              t_storet_pk_type,
     fraction            VARCHAR2(40),
     value_type          NUMBER(10),
     stat_type           NUMBER(10),
     dur_basis           NUMBER(10),
     temp_basis          NUMBER(10)
     );

   TYPE t_stvst_id_rec IS RECORD (
     trip_seq            NUMBER(10),
     station_is_nbr      NUMBER(8),
     visit_id            NUMBER(3)
     );

   TYPE t_stvst_id_type IS TABLE OF t_stvst_id_rec INDEX BY BINARY_INTEGER;

   TYPE t_stvst_id_key_type IS TABLE OF NUMBER(19) INDEX BY BINARY_INTEGER;

  -- Public function and procedure declarations

   FUNCTION get_org_info(p_org_id IN VARCHAR2) RETURN NUMBER;

   FUNCTION get_delimiter(p_imp_cfg_seq IN NUMBER) RETURN VARCHAR2;

   FUNCTION get_item_pos(p_sicc_seq IN NUMBER, p_pos_table OUT col_table_num_type,
                         p_default_table OUT col_table_type,p_col_cnt IN OUT NUMBER) RETURN NUMBER;

   FUNCTION check_translations(p_code IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_col_seq IN NUMBER,
                               p_trans_exist OUT BOOLEAN) RETURN VARCHAR2;

   PROCEDURE create_trans_record(p_code IN VARCHAR2,p_imp_cfg_seq IN NUMBER,p_col_seq IN NUMBER,p_fa_seq IN NUMBER);

   FUNCTION check_allowable_values(p_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_col_seq IN NUMBER,p_fa_seq IN NUMBER)
     RETURN NUMBER;

   FUNCTION validate_alval_item(p_item_name IN VARCHAR2, p_orig_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER,
                                p_col_seq IN NUMBER, p_fa_seq IN NUMBER) RETURN NUMBER;

   FUNCTION validate_storet_item(p_item_name IN VARCHAR2, p_orig_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER,
                                 p_col_seq IN NUMBER, p_parent_pk IN t_storet_pk_type, p_org_id IN VARCHAR2,
                                 p_line_nbr IN NUMBER, p_fa_seq IN NUMBER) RETURN t_storet_pk_type;

   FUNCTION get_date(p_column_name IN VARCHAR2,p_raw_date IN VARCHAR2,p_col_seq IN NUMBER,
                     p_cfg_seq IN NUMBER,p_fa_seq IN NUMBER) RETURN DATE;

   FUNCTION get_time(p_column_name IN VARCHAR2,p_raw_time IN VARCHAR2,p_col_seq IN NUMBER,
                     p_cfg_seq IN NUMBER,p_fa_seq IN NUMBER) RETURN DATE;

   FUNCTION check_gear_type(p_proc_is_nbr IN NUMBER, p_gear_is_nbr IN NUMBER, p_org_id IN VARCHAR2, p_fa_seq IN NUMBER)
     RETURN NUMBER;

   FUNCTION add_project(p_project_id IN VARCHAR2,p_org_id IN VARCHAR2, p_fa_seq IN NUMBER) RETURN NUMBER;

   FUNCTION get_analysis_code_seq(p_activity_seq IN NUMBER,p_analysis_code IN VARCHAR2,p_status IN OUT VARCHAR2,
                                  p_tsrlab_is_nbr IN NUMBER, p_tsrlab_org_id IN VARCHAR2,
                                  p_line_nbr IN NUMBER, p_fa_seq IN NUMBER) RETURN NUMBER;

   FUNCTION get_value_msr (p_result_text IN VARCHAR2, p_result_ok OUT BOOLEAN, p_fa_seq IN NUMBER) RETURN NUMBER;

   PROCEDURE import_results (p_org_id IN VARCHAR2, p_import_log_seq IN NUMBER,
                          p_imp_cfg_seq IN NUMBER,p_test_count IN NUMBER);


END Sim2_Import_Result_Pkg;
/
CREATE OR REPLACE PACKAGE BODY Sim2_Import_Result_Pkg IS

   v_line_cnt           NUMBER(10) := 0;
   v_activity_count     t_activity_count;
   v_acode_defs_rec     t_acode_defs_rec;
   v_acode_copy_rec     t_acode_defs_rec;
   v_org_is_nbr         NUMBER(8);
   v_import_log_seq     NUMBER(10);
   v_trip_qc            NUMBER(1) := 0;
   v_cur_trip_is_nbr    NUMBER(8) := NULL;
   --v_cur_proj_is_nbr    NUMBER(8) := NULL;

   v_last_visit_id      NUMBER(5) := 0;

   --v_skip_result        BOOLEAN := FALSE;

   --e_a   EXCEPTION;

/******************************************************************************************************/
FUNCTION get_org_info(p_org_id IN VARCHAR2) RETURN NUMBER IS

  v_org_is_nbr                NUMBER(8);

  CURSOR c_get_tsmorgan_nbr (p_cursor_org_id IN VARCHAR2) IS
    SELECT TSMORGAN_IS_NUMBER FROM TSMORGAN WHERE ORG_ID = RPAD(p_cursor_org_id,8);

BEGIN
  /* kms - function that gets the is_number for the organization running the import.  Called by main
     import procedure - import_results */

  OPEN c_get_tsmorgan_nbr (p_org_id);
  FETCH c_get_tsmorgan_nbr INTO v_org_is_nbr;
    IF c_get_tsmorgan_nbr%NOTFOUND THEN
      v_org_is_nbr := NULL;
      Sim2_Import_Pkg.write_error(v_import_log_seq, 'Failed to get organization is number.',NULL,NULL,NULL);
    END IF;
  CLOSE c_get_tsmorgan_nbr;

  RETURN v_org_is_nbr;

EXCEPTION
  WHEN OTHERS THEN
    RAISE_APPLICATION_ERROR(-20199 , 'Failed to get organization is number.');
    RETURN NULL;

END get_org_info;

/******************************************************************************************************/
FUNCTION get_delimiter(p_imp_cfg_seq IN NUMBER) RETURN VARCHAR2 IS

  v_delimiter          VARCHAR2(1);

  CURSOR c_get_imp_cfg(p_sicc_seq IN NUMBER) IS
    SELECT SICC_DELIMITER FROM SIM_IMP_CFG WHERE SICC_SEQ = p_sicc_seq;

BEGIN
  /* kms - function that retrieves the delimiter from the import configuration table.  Called by main
     import procedure - import_results. */

  OPEN c_get_imp_cfg(p_imp_cfg_seq);
  FETCH c_get_imp_cfg INTO v_delimiter;
    IF c_get_imp_cfg%NOTFOUND THEN
      v_delimiter := NULL;
    END IF;
  CLOSE c_get_imp_cfg;

  RETURN v_delimiter;

EXCEPTION

  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - Failed to get delimiter.', NULL, NULL, NULL);
    RETURN NULL;

END get_delimiter;

/******************************************************************************************************/
FUNCTION get_item_pos(p_sicc_seq IN NUMBER, p_pos_table OUT col_table_num_type,
                      p_default_table OUT col_table_type,p_col_cnt IN OUT NUMBER) RETURN NUMBER IS

  v_import_type_seq              NUMBER(10);

  e_bad_cfg                      EXCEPTION;

  CURSOR c_get_import_type(p_sicc_seq IN NUMBER) IS
    SELECT SICC_SID_SEQ FROM SIM_IMP_CFG WHERE SICC_SEQ = p_sicc_seq;

  CURSOR c_get_dflt_pos (p_type_seq IN NUMBER) IS
    SELECT SIC_DFLT_POS FROM SIM_IMP_COLS WHERE SIC_SID_SEQ = p_type_seq;

  CURSOR c_get_col_pos (p_sicc_seq IN NUMBER,p_dflt_pos IN NUMBER) IS
    SELECT SICDT_POS, SICDT_DEFAULT FROM SIM_IMP_CFG_DETS, SIM_IMP_COLS
      WHERE SIC_SEQ = SICDT_SIC_SEQ AND SICDT_SICC_SEQ = p_sicc_seq AND SIC_DFLT_POS = p_dflt_pos
      AND SICDT_INCLUDED IS NOT NULL;

BEGIN
  /* kms - function to populate the tables that will hold the import configuration information.
     This will populate tables that hold actual position information and default information
     indexed by the default column position.  It will also count the columns in order to insure
     that they match the import configuration.  Called by main import procedure - import_results. */

  /* get the import type in order to limit the columns selected to those that are for results
     and activities */
  OPEN c_get_import_type(p_sicc_seq);
  FETCH c_get_import_type INTO v_import_type_seq;
   IF c_get_import_type%NOTFOUND THEN
     v_import_type_seq := NULL;
     CLOSE c_get_import_type;
     RAISE e_bad_cfg;
   END IF;
  CLOSE c_get_import_type;

  p_col_cnt := 0;

  /* loop through import columns and assign actual position to a table (p_pos_table) indexed by
     default position */
  FOR v_dflt_pos IN c_get_dflt_pos(v_import_type_seq) LOOP

    /* get items' actual position and default value from configuration table */
    OPEN c_get_col_pos(p_sicc_seq,v_dflt_pos.sic_dflt_pos);
    FETCH c_get_col_pos INTO p_pos_table(v_dflt_pos.sic_dflt_pos), p_default_table(v_dflt_pos.sic_dflt_pos);
      IF c_get_col_pos%NOTFOUND THEN
        /* if the column has not been included in the import configuration, it's position and
           default will be null */
        p_pos_table(v_dflt_pos.sic_dflt_pos) := NULL;
        p_default_table(v_dflt_pos.sic_dflt_pos) := NULL;
      ELSE
        p_col_cnt := p_col_cnt + 1;
      END IF;
    CLOSE c_get_col_pos;

  END LOOP;

  RETURN 1;

EXCEPTION
  WHEN e_bad_cfg THEN
    RETURN 0;

  WHEN OTHERS THEN
    RAISE_APPLICATION_ERROR(-20103 , 'Failed to get column positions, item names, and column sequences.');

END get_item_pos;

/******************************************************************************************************/
FUNCTION get_instructions(p_imp_cfg_seq IN NUMBER, p_instruction_type IN VARCHAR2, p_fa_seq IN NUMBER)
  RETURN VARCHAR2 IS

  v_instr_choice         VARCHAR2(15);

  CURSOR c_get_cfg_instr(p_cfg_seq IN NUMBER, p_instr_type IN VARCHAR2) IS
    SELECT SIIC_CODE FROM SIM_IMP_CFG_INSTRS, SIM_IMP_INSTR_CHOICES, SIM_IMP_INSTRS
      WHERE SICI_SIIC_SEQ = SIIC_SEQ AND SIIC_SII_SEQ = SII_SEQ
      AND SII_CODE = p_instr_type AND SICI_SICC_SEQ = p_cfg_seq;

  CURSOR c_get_instr_default(p_instr_type IN VARCHAR2) IS
    SELECT SIIC_CODE FROM SIM_IMP_INSTR_CHOICES, SIM_IMP_CFG_INSTRS, SIM_IMP_INSTRS
      WHERE SICI_SIIC_SEQ = SIIC_SEQ AND SIIC_SII_SEQ = SII_SEQ
      AND SII_CODE = p_instr_type AND SIIC_DFLT_FLAG = 1;

BEGIN

  /* kms - get the instruction choice for this item */
  OPEN c_get_cfg_instr(p_imp_cfg_seq, p_instruction_type);
  FETCH c_get_cfg_instr INTO v_instr_choice;
    IF c_get_cfg_instr%NOTFOUND THEN

      /* no choice could be found, get default choice */
      OPEN c_get_instr_default(p_instruction_type);
      FETCH c_get_instr_default INTO v_instr_choice;
      CLOSE c_get_instr_default;

    END IF;
  CLOSE c_get_cfg_instr;

  RETURN v_instr_choice;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in get_instructions.',NULL,p_fa_seq,NULL);
    RETURN NULL;

END get_instructions;

/******************************************************************************************************/
FUNCTION GET_PROJ_IS_NBR (p_proj VARCHAR2, p_org_id VARCHAR2, p_fa_seq IN NUMBER) RETURN t_storet_pk_type IS

  v_proj_key             t_storet_pk_type;

  CURSOR c_proj_is_nbr (p_proj VARCHAR2, p_org_id VARCHAR2) IS
    SELECT TSMPROJ_IS_NUMBER, TSMPROJ_ORG_ID FROM TSMPROJ WHERE TSMPROJ_ORG_ID = RPAD(p_org_id,8) AND
      IDENTIFICATION_CD = RPAD(p_proj,8);

BEGIN
  /* kms - function that gets the project primary key using the processed project id - called by validate storet item*/
  OPEN c_proj_is_nbr (p_proj, p_org_id);
  FETCH c_proj_is_nbr INTO v_proj_key;
    IF c_proj_is_nbr%NOTFOUND THEN
       v_proj_key.is_number := NULL;
       v_proj_key.org_id := NULL;
       Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Project, '||p_proj||', could not be found in STORET.',NULL,p_fa_seq,NULL);
    END IF;
  CLOSE c_proj_is_nbr;

  RETURN v_proj_key;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in get project is number.',NULL,p_fa_seq,NULL);
    v_proj_key.is_number := NULL;
    v_proj_key.org_id := NULL;
    RETURN v_proj_key;

END GET_PROJ_IS_NBR;

/******************************************************************************************************/
FUNCTION get_lab_is_nbr(p_lab_id IN VARCHAR2, p_org_id IN VARCHAR2, p_fa_seq IN NUMBER)
  RETURN t_storet_pk_type IS

  v_tsrlab_key                   t_storet_pk_type;

  CURSOR c_find_lab(p_lab_id IN VARCHAR2, p_org_id IN VARCHAR2) IS
    SELECT TSRLAB_IS_NUMBER, TSRLAB_ORG_ID FROM TSRLAB WHERE ID_CODE = RPAD(p_lab_id, 8)
      AND TSRLAB_ORG_ID = RPAD(p_org_id, 8);

BEGIN
  /* kms - function that gets the lab primary key using the processed lab id - called by validate storet item */
  OPEN c_find_lab(p_lab_id, p_org_id);
  FETCH c_find_lab INTO v_tsrlab_key;
    IF c_find_lab%NOTFOUND THEN
      v_tsrlab_key.is_number := NULL;
      v_tsrlab_key.org_id := NULL;
      Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Lab ID - '||p_lab_id||', could not be found in STORET.',NULL,p_fa_seq,NULL);
    END IF;
  CLOSE c_find_lab;

  RETURN v_tsrlab_key;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in get lab is number.',NULL,p_fa_seq,NULL);
    v_tsrlab_key.is_number := NULL;
    v_tsrlab_key.org_id := NULL;
    RETURN v_tsrlab_key;

END get_lab_is_nbr;

/******************************************************************************************************/
FUNCTION GET_GEAR_IS_NBR (p_gear IN VARCHAR2, p_fa_seq IN NUMBER) RETURN t_storet_pk_type IS

  v_gear_key             t_storet_pk_type;

  CURSOR c_gear_is_nbr (p_gear VARCHAR2) IS
     SELECT TSRFLDGR_IS_NUMBER, TSRFLDGR_ORG_ID FROM TSRFLDGR WHERE ID_CODE = RPAD(p_gear,10);

BEGIN
  /* kms - function that gets the gear primary key using the processed gear id - called by validate storet item */
   OPEN c_gear_is_nbr (p_gear);
   FETCH c_gear_is_nbr INTO v_gear_key;
     IF c_gear_is_nbr%NOTFOUND THEN
        v_gear_key.is_number := NULL;
        v_gear_key.org_id := NULL;
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Gear, '||p_gear||', could not be found in STORET.',NULL,p_fa_seq,NULL);
     END IF;
   CLOSE c_gear_is_nbr;

   RETURN v_gear_key;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in get gear is number.',NULL,p_fa_seq,NULL);
    v_gear_key.is_number := NULL;
    v_gear_key.org_id := NULL;
    RETURN v_gear_key;

END get_gear_is_nbr;

/******************************************************************************************************/
FUNCTION get_gear_cfg_is_nbr (p_gear_cfg IN VARCHAR2, p_gear_pk IN t_storet_pk_type,
                              p_org_id IN VARCHAR2, p_fa_seq IN NUMBER) RETURN t_storet_pk_type IS

  v_gear_cfg_key             t_storet_pk_type;

  CURSOR c_gear_cfg_is_nbr (p_gear_cfg IN VARCHAR2, p_gear_pk IN t_storet_pk_type, p_org_id IN VARCHAR2) IS
     SELECT TSRGRCFG_IS_NUMBER, TSRGRCFG_ORG_ID FROM TSRGRCFG
     WHERE ID_CODE = RPAD(p_gear_cfg,10) AND TSRGRCFG_ORG_ID = RPAD(p_org_id,8)
     AND TSRFLDGR_IS_NUMBER = p_gear_pk.is_number AND TSRFLDGR_ORG_ID = RPAD(p_gear_pk.org_id,8);

BEGIN
  /* kms (5/21/01) - function that gets the gear config primary key using the processed gear
     config id.  Called by validate storet item */
   OPEN c_gear_cfg_is_nbr (p_gear_cfg,p_gear_pk,p_org_id);
   FETCH c_gear_cfg_is_nbr INTO v_gear_cfg_key;
     IF c_gear_cfg_is_nbr%NOTFOUND THEN
        /* kms - gear config either does not exist or is not right for this gear */
        v_gear_cfg_key.is_number := NULL;
        v_gear_cfg_key.org_id := NULL;
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Gear Configuration, '||p_gear_cfg||', does not exist for the gear on this activity.',NULL,p_fa_seq,NULL);
     END IF;
   CLOSE c_gear_cfg_is_nbr;

   RETURN v_gear_cfg_key;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in get gear cfg is number.',NULL,p_fa_seq,NULL);
    v_gear_cfg_key.is_number := NULL;
    v_gear_cfg_key.org_id := NULL;
    RETURN v_gear_cfg_key;

END get_gear_cfg_is_nbr;

/******************************************************************************************************/
FUNCTION get_sample_pres_is_nbr (p_proc IN VARCHAR2, p_org_id IN VARCHAR2, p_fa_seq IN NUMBER)
   RETURN t_storet_pk_type IS

   v_proc_key                 t_storet_pk_type;

   CURSOR c_proc_is_nbr (p_proc VARCHAR2, p_org_id VARCHAR2) IS
     SELECT TSRSDP_IS_NUMBER, TSRSDP_ORG_ID FROM TSRSDP WHERE TSRSDP_ORG_ID = RPAD(p_org_id,8)
       AND ID_CODE = RPAD(p_proc,10);

BEGIN
  /* kms - function that gets the sample preservation primary key using the processed id_code.
     Called by validate storet item */
   OPEN c_proc_is_nbr (p_proc, p_org_id );
   FETCH c_proc_is_nbr INTO v_proc_key;
     IF c_proc_is_nbr%NOTFOUND THEN
        v_proc_key.is_number := NULL;
        v_proc_key.org_id := NULL;
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Sample Preservation, '||p_proc||', could not be found in STORET.',NULL,p_fa_seq,NULL);
     END IF;
   CLOSE c_proc_is_nbr;

   RETURN v_proc_key;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in get sample pres is number.',NULL,p_fa_seq,NULL);
    v_proc_key.is_number := NULL;
    v_proc_key.org_id := NULL;
    RETURN v_proc_key;

END get_sample_pres_is_nbr;

/******************************************************************************************************/
FUNCTION get_group_is_nbr(p_group_id IN VARCHAR2, p_org_id IN VARCHAR2,
                          p_fa_seq IN NUMBER) RETURN t_storet_pk_type IS

  v_group_key             t_storet_pk_type;

  CURSOR c_group_is_nbr (p_group IN VARCHAR2) IS
     SELECT TSRCHGRP_IS_NUMBER, TSRCHGRP_ORG_ID FROM TSRCHGRP
       WHERE ID_CODE = RPAD(p_group,8) AND TSRCHGRP_ORG_ID = RPAD(p_org_id,8);

BEGIN
  /* kms - function that gets the characteristic group primary key using the processed group id - called by validate storet item */
   OPEN c_group_is_nbr (p_group_id);
   FETCH c_group_is_nbr INTO v_group_key;
     IF c_group_is_nbr%NOTFOUND THEN
        v_group_key.is_number := NULL;
        v_group_key.org_id := NULL;
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Characteristic Group ID, '||p_group_id||', could not be found in STORET.',NULL,p_fa_seq,NULL);
     END IF;
   CLOSE c_group_is_nbr;

   RETURN v_group_key;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in get group is number.',NULL,p_fa_seq,NULL);
    v_group_key.is_number := NULL;
    v_group_key.org_id := NULL;
    RETURN v_group_key;

END get_group_is_nbr;

/******************************************************************************************************/
FUNCTION get_row_is_nbr(p_row_id IN VARCHAR2, p_group_key IN t_storet_pk_type, p_org_id IN VARCHAR2,
                        p_fa_seq IN NUMBER) RETURN t_storet_pk_type IS

  v_row_id_key             t_storet_pk_type;
  v_comp_cd                VARCHAR2(1);

  CURSOR c_get_row_id_is_nbr (p_row_id IN VARCHAR2, p_group_pk IN t_storet_pk_type) IS
     SELECT TSRCHDEF_IS_NUMBER, TSRCHDEF_ORG_ID, COMP_IND_CD FROM TSRCHDEF, TSRCCGA
       WHERE TSRCHDEF.TSRCCGA_IS_NUMBER = TSRCCGA.TSRCCGA_IS_NUMBER
       AND TSRCHDEF.TSRCCGA_ORG_ID = TSRCCGA.TSRCCGA_ORG_ID
       AND ROW_ID = RPAD(p_row_id,8) AND TSRCCGA.TSRCHGRP_IS_NUMBER = p_group_pk.is_number
       AND TSRCCGA.TSRCHGRP_ORG_ID = RPAD(p_group_pk.org_id,8);

BEGIN
  /* kms - function that gets the characteristic row primary key using the processed row id - called by validate storet item */
   OPEN c_get_row_id_is_nbr (p_row_id, p_group_key);
   FETCH c_get_row_id_is_nbr INTO v_row_id_key.is_number,v_row_id_key.org_id,v_comp_cd;
     IF c_get_row_id_is_nbr%NOTFOUND THEN
        v_row_id_key.is_number := NULL;
        v_row_id_key.org_id := NULL;
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Characteristic Row ID, '||p_row_id||', could not be found in STORET or is not a row in the provided characteristic group.',NULL,p_fa_seq,NULL);
     ELSE
       IF v_comp_cd = 'N' THEN
        v_row_id_key.is_number := NULL;
        v_row_id_key.org_id := NULL;
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Characteristic Row ID, '||p_row_id||', is not a complete row.',NULL,p_fa_seq,NULL);
       END IF;
     END IF;
   CLOSE c_get_row_id_is_nbr;

   RETURN v_row_id_key;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in get row_id is number.',NULL,p_fa_seq,NULL);
    v_row_id_key.is_number := NULL;
    v_row_id_key.org_id := NULL;
    RETURN v_row_id_key;

END get_row_is_nbr;

/******************************************************************************************************/
FUNCTION GET_PROCEDURE_IS_NBR (p_proc IN VARCHAR2, p_org_id IN VARCHAR2, p_fa_seq IN NUMBER)
   RETURN t_storet_pk_type IS

   v_proc_key                 t_storet_pk_type;

   CURSOR c_proc_is_nbr (p_proc VARCHAR2, p_org_id VARCHAR2) IS
     SELECT TSRFLDPR_IS_NUMBER, TSRFLDPR_ORG_ID FROM TSRFLDPR WHERE TSRFLDPR_ORG_ID = RPAD(p_org_id,8)
       AND ID_CODE = RPAD(p_proc,10);

BEGIN
  /* kms - function that gets the field procedure primary key using the processed procedure id - called by validate storet item */
   OPEN c_proc_is_nbr (p_proc, p_org_id );
   FETCH c_proc_is_nbr INTO v_proc_key;
     IF c_proc_is_nbr%NOTFOUND THEN
        v_proc_key.is_number := NULL;
        v_proc_key.org_id := NULL;
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Sample Collection Procedure, '||p_proc||', could not be found in STORET.',NULL,p_fa_seq,NULL);
     END IF;
   CLOSE c_proc_is_nbr;

   RETURN v_proc_key;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in get field procedure is number.',NULL,p_fa_seq,NULL);
    v_proc_key.is_number := NULL;
    v_proc_key.org_id := NULL;
    RETURN v_proc_key;

END get_procedure_is_nbr;

/******************************************************************************************************/
FUNCTION get_station_is_nbr(p_station_id IN VARCHAR2, p_org_id IN VARCHAR2, p_line_cnt IN NUMBER, p_fa_seq IN NUMBER)
  RETURN t_storet_pk_type IS

  CURSOR c_find_station (p_station_id IN VARCHAR2, p_org_id IN VARCHAR2) IS
    SELECT TSMSTATN_IS_NUMBER, TSMSTATN_ORG_ID, D_COMPLETE_FLAG FROM TSMSTATN
      WHERE IDENTIFICATION_CD = RPAD(p_station_id,15) AND TSMSTATN_ORG_ID = RPAD(p_org_id,8);

  v_station_key   t_storet_pk_type;
  v_complete_flag VARCHAR2(1);
  
BEGIN
  /* kms - function that gets the station primary key using the processed station id - called by validate storet item */
  OPEN c_find_station (p_station_id ,p_org_id);
  FETCH c_find_station INTO v_station_key.is_number, v_station_key.org_id, v_complete_flag;
    IF c_find_station%NOTFOUND THEN
      v_station_key.is_number := NULL;
      v_station_key.org_id := NULL;
      Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Station, '||p_station_id||', could not be found in STORET.',NULL,p_fa_seq,NULL);
    ELSIF v_complete_flag = 'N' THEN 
      v_station_key.is_number := NULL;
      v_station_key.org_id := 'XXX';
      Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Station, '||p_station_id||', is incomplete.',NULL,p_fa_seq,NULL);
    END IF;
  CLOSE c_find_station;

  RETURN v_station_key;

EXCEPTION
   WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - PROC GET_STATION_IS_NBR, line: '||p_line_cnt,NULL,p_fa_seq,NULL);
      v_station_key.is_number := NULL;
      v_station_key.org_id := NULL;
      RETURN v_station_key;

END get_station_is_nbr;

/******************************************************************************************************/
FUNCTION get_uom_is_nbr (p_uom_name IN VARCHAR2, p_fa_seq IN NUMBER) RETURN t_storet_pk_type IS

   TYPE t_uom_record IS RECORD (
      uom_is_nbr        NUMBER(8),
      uom_org_id        VARCHAR2(8),
      uom_name          VARCHAR2(10)
      );

   v_uom_info              t_uom_record;
   v_uom_key               t_storet_pk_type;

   e_unk_uom               EXCEPTION;
   e_uom_sp                EXCEPTION;

   CURSOR c_get_is_nbr (p_uom_name IN VARCHAR2) IS
     SELECT TSRUOM_IS_NUMBER, TSRUOM_ORG_ID, SHORT_FORM_NAME FROM TSRUOM
       WHERE UPPER(SHORT_FORM_NAME) = UPPER(RPAD(p_uom_name,10));

BEGIN
  /* kms - function that gets the uom primary key using the processed uom name - called by validate storet item
     Also checks the case of the unit of measure name and issues a warning if it is not correct */

   /* get tsruom key for unit of measure */
   OPEN c_get_is_nbr(p_uom_name);
   FETCH c_get_is_nbr INTO v_uom_info;
     IF c_get_is_nbr%NOTFOUND THEN

       /* if uom could not be found then raise an exception to write error to error log, will change status in main procedure */
       v_uom_info.uom_is_nbr := NULL;
       v_uom_info.uom_org_id := NULL;
       CLOSE c_get_is_nbr;
       RAISE e_unk_uom;

     ELSE

       IF RPAD(v_uom_info.uom_name,10) != RPAD(p_uom_name,10) THEN
         CLOSE c_get_is_nbr;
         RAISE e_uom_sp;
       END IF;

     END IF;
   CLOSE c_get_is_nbr;

   v_uom_key.is_number := v_uom_info.uom_is_nbr;
   v_uom_key.org_id := v_uom_info.uom_org_id;

   RETURN v_uom_key;

EXCEPTION
   WHEN e_unk_uom THEN
     Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, 'Line '||v_line_cnt||': Unknown Unit of Measure: '||p_uom_name,NULL,p_fa_seq,NULL);
     RETURN v_uom_key;

   WHEN e_uom_sp THEN
     Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, 'Line '||v_line_cnt||': Warning: Unit of Measure misspelled: '||p_uom_name||' - check capitalization.',NULL,p_fa_seq,NULL);
     RETURN v_uom_key;

   WHEN OTHERS THEN
     Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in get uom is number.',NULL,p_fa_seq,NULL);
     v_uom_key.is_number := NULL;
     v_uom_key.org_id := NULL;
     RETURN v_uom_key;

END get_uom_is_nbr;

/******************************************************************************************************/
FUNCTION get_personnel_is_nbr(p_contact_id IN VARCHAR2, p_orig_item IN VARCHAR2,p_fa_seq IN NUMBER)
  RETURN t_storet_pk_type IS

  v_first_name                  VARCHAR2(30);
  v_last_name                   VARCHAR2(40);
  v_tsmpersn_key                t_storet_pk_type;
  v_contact_seq                 NUMBER(10);

  CURSOR c_find_contact (p_first_name IN VARCHAR2, p_last_name IN VARCHAR2) IS
    SELECT TSMPERSN_IS_NUMBER, TSMPERSN_ORG_ID FROM TSMPERSN WHERE RPAD(FIRST_NAME,30) = RPAD(p_first_name,30)
      AND RPAD(LAST_NAME,40) = RPAD(p_last_name,40);


BEGIN
  /* kms - function that gets personnel primary key using the processed personnel name.  Expects to get last
     name first and that it is separated from the first name with a space.  If personnel cannot be found, this
     function will create it in the STORET table.  Called by validate_storet_item. */

  /* divide contact id into first and last names
     - assumes first space is break between first and last names */
  v_last_name := RTRIM(SUBSTR(p_contact_id,1,INSTR(p_contact_id,' ')));
  v_first_name := LTRIM(SUBSTR(p_contact_id,INSTR(p_contact_id,' ')));

  OPEN c_find_contact(v_first_name,v_last_name);
  FETCH c_find_contact INTO v_tsmpersn_key;
    IF c_find_contact%NOTFOUND THEN
      v_tsmpersn_key.is_number := NULL;
      v_tsmpersn_key.org_id := NULL;
      Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Contact, '||p_contact_id||', could not be found in STORET.',NULL,p_fa_seq,NULL);
      /* create personnel record in STORET */
    END IF;
  CLOSE c_find_contact;

RETURN v_tsmpersn_key;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in get personnel is nbr.',NULL,p_fa_seq,NULL);
    v_tsmpersn_key.is_number := NULL;
    v_tsmpersn_key.org_id := NULL;
    RETURN v_tsmpersn_key;

END get_personnel_is_nbr;

/******************************************************************************************************/
FUNCTION get_char_is_nbr (p_char_name IN VARCHAR2, p_fa_seq IN NUMBER) RETURN t_storet_pk_type IS

  v_char_key             t_storet_pk_type;

  CURSOR c_get_is_nbr(p_char_name IN VARCHAR2) IS
    SELECT TSRCHAR_IS_NUMBER, TSRCHAR_ORG_ID FROM TSRCHAR WHERE DISPLAY_NAME = p_char_name;

BEGIN
  /* kms - function that checks the characteristic in STORET using the processed display name.
     Called by validate storet item. */

  OPEN c_get_is_nbr(p_char_name);
  FETCH c_get_is_nbr INTO v_char_key;
    IF c_get_is_nbr%NOTFOUND THEN
      v_char_key.is_number := NULL;
      v_char_key.org_id := NULL;
      Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Characteristic, '||p_char_name||', could not be found in STORET.',NULL,p_fa_seq,NULL);
    END IF;
  CLOSE c_get_is_nbr;

  RETURN v_char_key;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - getting characteristic is number',NULL,p_fa_seq,NULL);
    v_char_key.is_number := NULL;
    v_char_key.org_id := NULL;
    RETURN v_char_key;

END get_char_is_nbr;

/******************************************************************************************************/
FUNCTION get_analproc_is_nbr (p_procedure_source_id IN VARCHAR2, p_org_id IN VARCHAR2,
                              p_fa_seq IN NUMBER) RETURN t_storet_pk_type IS

  v_proc_key               t_storet_pk_type;
  v_procedure_id           VARCHAR2(15);
  --v_source                 VARCHAR2(12);
  v_org_is_nbr             NUMBER(8);
  v_temp_is_nbr            NUMBER(8);
  v_type_cd                VARCHAR2(2);
  v_success                NUMBER(1);

  e_bad_proc               EXCEPTION;

/* kms temp fix for superfund/multiple column problems
  CURSOR c_get_is_nbr(p_procedure_id IN VARCHAR2, p_source IN VARCHAR2) IS
    SELECT TSRANLPR_IS_NUMBER, TSRANLPR_ORG_ID FROM TSRANLPR WHERE PROCEDURE_ID = RPAD(p_procedure_id, 15)
    AND SOURCE_ACR = RPAD(p_source,12);
*/
   CURSOR c_get_is_nbr(p_procedure_id IN VARCHAR2) IS
    SELECT TSRANLPR_IS_NUMBER, TSRANLPR_ORG_ID FROM TSRANLPR WHERE PROCEDURE_ID = RPAD(p_procedure_id, 15);

  CURSOR c_check_adopt(p_proc_is_nbr IN NUMBER, p_proc_org_id IN VARCHAR2, p_org_is_nbr IN NUMBER) IS
    SELECT TSMORGAN_IS_NUMBER FROM TSROAPA WHERE TSRANLPR_IS_NUMBER = p_proc_is_nbr
      AND TSRANLPR_ORG_ID = RPAD(p_proc_org_id,8) AND TSMORGAN_IS_NUMBER = p_org_is_nbr;

BEGIN
  /* kms - function that gets the analysis procedure primary key using the processed procedure id.  This
     should consist of the procedure id followed by the source acronym, separated by a '\'.  Called from
     validate_storet_item */

  --kms ****Changed this function to use just the procedure id for now. 3/12/01

  /* divide id into procedure id and owner */
  --kms  v_procedure_id := SUBSTR(p_procedure_source_id,1,INSTR(p_procedure_source_id,'\')-1);
  --KMS  v_source := SUBSTR(p_procedure_source_id,INSTR(p_procedure_source_id,'\')+1);
  v_procedure_id := p_procedure_source_id;

  /* get prep is number for this id and owner */
  OPEN c_get_is_nbr(v_procedure_id);
  FETCH c_get_is_nbr INTO v_proc_key;
    IF c_get_is_nbr%NOTFOUND THEN
      v_proc_key.is_number := NULL;
      v_proc_key.org_id := NULL;
      RAISE e_bad_proc;
    END IF;
  CLOSE c_get_is_nbr;

  /* get the org is number */
   v_org_is_nbr := get_org_info(p_org_id);

  /* make sure analysis procedure has been adopted by this org */
  IF v_proc_key.is_number IS NOT NULL THEN
    OPEN c_check_adopt(v_proc_key.is_number, v_proc_key.org_id, v_org_is_nbr);
    FETCH c_check_adopt INTO v_temp_is_nbr;
      IF c_check_adopt%NOTFOUND THEN
        /* kms - should only adopt the procedure if it's national or if it belongs
           to this org */
        IF v_proc_key.org_id = RPAD(p_org_id,8) OR v_proc_key.org_id = 'ZZZZZZZZ' THEN
          /* kms - set adoption type code to indicate whether or not the analysis procedure
             is owned by this organization */
          IF v_proc_key.org_id = 'ZZZZZZZZ' THEN
            v_type_cd := 'AD';
          ELSE
            v_type_cd := 'OW';
          END IF;

          /* ADDED BY SNG 3/5/01 - ADOPT THE PROCEDURE ...) */
          INSERT INTO TSROAPA (
            TSMORGAN_IS_NUMBER,
            TSRANLPR_IS_NUMBER,
            TSRANLPR_ORG_ID,
            TYPE_CODE,
            SUSPEND_INDICATOR,
            D_USERID_CODE,
            D_LAST_UPDATE_TS )
          VALUES (
            v_org_is_nbr,
            v_proc_key.is_number,
            v_proc_key.org_id,
            v_type_cd,
            'N',
            RPAD(SUBSTR(USER,1,8),8),
            TRUNC(SYSDATE)
            );
        ELSE
          /* kms - procedure exists, but belongs to another organization, do not allow it to be
             used.  Write error message */
          Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': The analysis procedure '||p_procedure_source_id||', is not valid: it is not a national procedure and does not belong to this organization.',NULL,p_fa_seq,NULL);
          v_proc_key.is_number := NULL;
          v_proc_key.org_id := NULL;
        END IF;
      END IF;
    CLOSE c_check_adopt;
  END IF;

  RETURN v_proc_key;

EXCEPTION
  WHEN e_bad_proc THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Unable to find analysis procedure '||p_procedure_source_id||' in STORET.',NULL,p_fa_seq,NULL);
    RETURN v_proc_key;

  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - getting analysis procedure is number',NULL,p_fa_seq,NULL);
    v_proc_key.is_number := NULL;
    v_proc_key.org_id := NULL;
    RETURN v_proc_key;

END get_analproc_is_nbr;

/******************************************************************************************************/
FUNCTION check_proc_char_assoc (p_procedure_key IN t_storet_pk_type, p_char_is_nbr IN NUMBER,
                                p_char_org_id IN VARCHAR2, p_org_is_nbr IN NUMBER,
                                p_fa_seq IN NUMBER) RETURN NUMBER IS

  v_return_value                NUMBER(1);
  v_temp_is_nbr                 NUMBER(8);

  CURSOR c_check_assoc(p_procedure_is_nbr IN NUMBER, p_procedure_org_id IN VARCHAR2,
                       p_char_is_nbr IN NUMBER, p_char_org_id IN VARCHAR2, p_org_is_nbr IN NUMBER) IS
    SELECT TSMORGAN_IS_NUMBER FROM TSROAPCA WHERE TSRANLPR_IS_NUMBER = p_procedure_is_nbr
      AND TSRCHAR_IS_NUMBER = p_char_is_nbr AND TSRANLPR_ORG_ID = p_procedure_org_id
      AND TSRCHAR_ORG_ID = p_char_org_id AND TSMORGAN_IS_NUMBER = p_org_is_nbr;

BEGIN
  /* kms - function that makes sure the analysis procedure has been associated with the characteristic.
     If they have not already been associated, this function will make the association. Called from the
     main import results procedure. */

  OPEN c_check_assoc(p_procedure_key.is_number, p_procedure_key.org_id, p_char_is_nbr, p_char_org_id, p_org_is_nbr);
  FETCH c_check_assoc INTO v_temp_is_nbr;
    IF c_check_assoc%NOTFOUND THEN

      /* make the association  SNG 3/4/01 */
      INSERT INTO TSROAPCA (
        TSRCHAR_IS_NUMBER,
        TSRCHAR_ORG_ID,
        TSMORGAN_IS_NUMBER,
        TSRANLPR_IS_NUMBER,
        TSRANLPR_ORG_ID,
        D_USERID_CODE,
        D_LAST_UPDATE_TS)
     VALUES (
        p_char_is_nbr,
        p_char_org_id,
        p_org_is_nbr,
        p_procedure_key.is_number,
        p_procedure_key.org_id,
        RPAD(SUBSTR(USER,1,8),8),
        TRUNC(SYSDATE)
        );

      v_return_value := 1;

    END IF;
  CLOSE c_check_assoc;

  RETURN v_return_value;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - checking analysis procedure characteristic association',NULL,p_fa_seq,NULL);
    RETURN 0;

END check_proc_char_assoc;

/******************************************************************************************************/
FUNCTION get_labprep_is_nbr (p_preparation_source_id IN VARCHAR2, p_org_id IN VARCHAR2,
                             p_fa_seq IN NUMBER) RETURN t_storet_pk_type IS

  v_prep_key               t_storet_pk_type;
  v_preparation_id         VARCHAR2(15);
  v_source                 VARCHAR2(12);
  v_type_code              VARCHAR2(3);
  v_org_is_nbr             NUMBER(8);
  v_success                NUMBER(1);

  e_bad_prep               EXCEPTION;

  CURSOR c_get_is_nbr(p_preparation_id IN VARCHAR2) IS --kms, p_source IN VARCHAR2) IS
    SELECT TSRLSPP_IS_NUMBER, TSRLSPP_ORG_ID FROM TSRLSPP WHERE PREPARATION_ID = RPAD(p_preparation_id, 15)
      ;--kms AND SOURCE_ACR = RPAD(p_source,12);

  CURSOR c_check_adopt(p_prep_is_nbr IN NUMBER, p_prep_org_id IN VARCHAR2, p_org_is_nbr IN NUMBER) IS
    SELECT TYPE_CODE FROM TSROLSPA WHERE TSRLSPP_IS_NUMBER = p_prep_is_nbr AND TSRLSPP_ORG_ID = RPAD(p_prep_org_id,8)
      AND TSMORGAN_IS_NUMBER = p_org_is_nbr;

BEGIN
  /* kms - function that gets the lab prep procedure primary key using the processed procedure id.  This
     should consist of the preparation id followed by the source acronym, separated by a '\'.  Called from
     validate_storet_item */

  /* divide id into prep id and owner */
  v_preparation_id := p_preparation_source_id; --kms SUBSTR(p_preparation_source_id,1,INSTR(p_preparation_source_id,'\')-1);
  --kms  v_source := SUBSTR(p_preparation_source_id,INSTR(p_preparation_source_id,'\')+1);

  /* get prep is number for this id and source */
  OPEN c_get_is_nbr(v_preparation_id);--kms ,v_source);
  FETCH c_get_is_nbr INTO v_prep_key;
    IF c_get_is_nbr%NOTFOUND THEN
      v_prep_key.is_number := NULL;
      v_prep_key.org_id := NULL;
      RAISE e_bad_prep;
    END IF;
  CLOSE c_get_is_nbr;

  /* get the org is number */
  v_org_is_nbr := get_org_info(p_org_id);

  /* make sure prep procedure has been adopted by this org */
  IF v_prep_key.is_number IS NOT NULL THEN
    OPEN c_check_adopt(v_prep_key.is_number, v_prep_key.org_id, v_org_is_nbr);
    FETCH c_check_adopt INTO v_type_code;
      IF c_check_adopt%NOTFOUND THEN
        /* if it hasn't been adopted, adopt */

        INSERT INTO TSROLSPA (
          TSMORGAN_IS_NUMBER,
          TSRLSPP_IS_NUMBER,
          TSRLSPP_ORG_ID,
          TYPE_CODE,
          D_USERID_CODE,
          D_LAST_UPDATE_TS
          )
        VALUES (
          v_org_is_nbr,
          v_prep_key.is_number,
          v_prep_key.org_id,
          'AD',
          USER,
          SYSDATE
          );

      END IF;
    CLOSE c_check_adopt;
  END IF;

  RETURN v_prep_key;

EXCEPTION
  WHEN e_bad_prep THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Unable to find lab preparation procedure '||p_preparation_source_id||' in STORET.',NULL,p_fa_seq,NULL);
    RETURN v_prep_key;

  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - getting lab prep is number',NULL,p_fa_seq,NULL);
    v_prep_key.is_number := NULL;
    v_prep_key.org_id := NULL;
    RETURN v_prep_key;

END get_labprep_is_nbr;

/******************************************************************************************************/
FUNCTION check_translations(p_code IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_col_seq IN NUMBER,
                            p_trans_exist OUT BOOLEAN) RETURN VARCHAR2 IS

  v_storet_value             VARCHAR2(80);

  CURSOR c_get_storet_code(p_code IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_col_seq IN NUMBER) IS
    SELECT st_storet_code FROM SIM_TRANSLATIONS, SIM_IMP_CFG_DETS WHERE st_user_code = p_code
      AND st_sicdt_seq = sicdt_seq AND sicdt_sicc_seq = p_imp_cfg_seq
      AND sicdt_sic_seq = p_col_seq;

BEGIN
  /* kms - function that takes the value from the text file for a given item and checks
     the translation table to see if it needs to be swapped with another value. */
  
  /* SEF-10/9/01 Since SIM_TRANSLATIONS.st_storet_code is only 80 chars, translation is not available in cases
   where we're looking at a long char string */
    IF length(p_code) > 80 then
       RETURN P_code;
    END IF;     
  /* check translations table to see if the code should be swapped with another value */
  OPEN c_get_storet_code(p_code,p_imp_cfg_seq,p_col_seq);
  FETCH c_get_storet_code INTO v_storet_value;
    IF c_get_storet_code%NOTFOUND THEN

      /* if not in translations table return original value - will check elsewhere for sequences */
      v_storet_value := p_code;
      p_trans_exist := FALSE;

    ELSIF c_get_storet_code%FOUND THEN
      /* if the code is in the translations table, flag so it's not recreated */
      p_trans_exist := TRUE;
      /*
      IF v_storet_value IS NULL THEN
        v_storet_value := p_code;
      END IF;
      */
    END IF;
  CLOSE c_get_storet_code;

  RETURN v_storet_value;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in check translations',NULL,NULL,NULL);
    RETURN NULL;

END check_translations;

/******************************************************************************************************/
PROCEDURE create_trans_record(p_code IN VARCHAR2,p_imp_cfg_seq IN NUMBER,p_col_seq IN NUMBER,
                              p_fa_seq IN NUMBER) IS

  v_st_seq          NUMBER(10);
  v_sicdt_seq       NUMBER(10);
  v_desc            VARCHAR2(80);

  e_imp_cfg_dets    EXCEPTION;
  e_trans_error     EXCEPTION;

  CURSOR c_get_cfg_det_seq(p_imp_cfg_seq IN NUMBER, p_col_seq IN NUMBER) IS
    SELECT SICDT_SEQ FROM SIM_IMP_CFG_DETS WHERE SICDT_SICC_SEQ = p_imp_cfg_seq
      AND SICDT_SIC_SEQ = p_col_seq;

  CURSOR c_get_st_seq IS
    SELECT sim_translations_seq.NEXTVAL FROM DUAL;

BEGIN

  /* kms - if it has been determined that value won't work for STORET, that a translation
     record does not already exist, and that the user wants to autocreate them, this function
     will create half of a translation record.  The remaining half will need to be filled in
     by the user after the import has completed. */

  OPEN c_get_cfg_det_seq(p_imp_cfg_seq, p_col_seq);
  FETCH c_get_cfg_det_seq INTO v_sicdt_seq;
    IF c_get_cfg_det_seq%NOTFOUND THEN
      v_sicdt_seq := NULL;
      RAISE e_imp_cfg_dets;
    END IF;
  CLOSE c_get_cfg_det_seq;

  /* get the sequence for the new record in translations table */
  OPEN c_get_st_seq;
  FETCH c_get_st_seq INTO v_st_seq;
    IF c_get_st_seq%NOTFOUND THEN
      v_st_seq := NULL;
      RAISE e_trans_error;
    END IF;
  CLOSE c_get_st_seq;

  INSERT INTO SIM_TRANSLATIONS (
    ST_SEQ,
    ST_SICDT_SEQ,
    ST_USER_CODE,
    ST_STORET_CODE
    )
  VALUES (
    v_st_seq,                  /* st_seq */
    v_sicdt_seq,               /* st_sicdt_seq */
    p_code,                    /* st_user_code */
    NULL                       /* st_storet_code */
    );

    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Created translation record for '||p_code,NULL,p_fa_seq,NULL);


EXCEPTION

  WHEN e_imp_cfg_dets THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Unable to find import configuration - stopped import',NULL,p_fa_seq,NULL);
    RAISE_APPLICATION_ERROR(-20101,'Unable to find import configuration');

  WHEN e_trans_error THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Unable to create new record in translations table - error getting sequence',NULL,p_fa_seq,NULL);

  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - adding record to translations table.',NULL,p_fa_seq,NULL);

END create_trans_record;

/******************************************************************************************************/
FUNCTION check_allowable_values(p_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_col_seq IN NUMBER,p_fa_seq IN NUMBER)
  RETURN NUMBER IS

  v_sia_seq         NUMBER(10);

  CURSOR c_check_allowable_val(p_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_col_seq IN NUMBER) IS
    SELECT SIA_SEQ FROM SIM_IMP_ALLOW_VALUES, SIM_IMP_COLS, SIM_IMP_CFG_DETS
      WHERE SIA_SIC_SEQ = SIC_SEQ AND SICDT_SICC_SEQ = p_imp_cfg_seq
      AND SICDT_SIC_SEQ = SIC_SEQ AND SIC_SEQ = p_col_seq AND SIA_ITEM = p_item;

BEGIN
  /* kms - function called for items that are listed as allowable values in SIM.  Checks
     processed item to make sure that it is one of the allowable values for that item and
     returns the sequence for that allowable value. */

  /* make sure item is an allowable value */
  OPEN c_check_allowable_val(p_item, p_imp_cfg_seq, p_col_seq);
  FETCH c_check_allowable_val INTO v_sia_seq;
    IF c_check_allowable_val%NOTFOUND THEN

      /* if not an allowable value, return null - will write error message, change status of activity */
      v_sia_seq := NULL;

    END IF;
  CLOSE c_check_allowable_val;

  RETURN v_sia_seq;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in check allowable values',NULL,p_fa_seq,NULL);
    RETURN NULL;

END check_allowable_values;

/******************************************************************************************************/
FUNCTION validate_alval_item(p_item_name IN VARCHAR2, p_orig_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER,
                             p_col_seq IN NUMBER, p_fa_seq IN NUMBER) RETURN NUMBER IS

   v_converted_item           VARCHAR2(80);
   v_item_sia_seq             NUMBER(10);
   v_trans_exist              BOOLEAN := TRUE;
   v_instr_code               VARCHAR2(15);

BEGIN
  /* kms - function to look in the translations and allowable values tables for items that are
     hard coded values in STORET */

  /* look in translations table to see if it needs to swapped - will return original item or translation */
  v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist);

  IF v_converted_item IS NOT NULL THEN

    /* look in allowable values */
    v_item_sia_seq := check_allowable_values(v_converted_item, p_imp_cfg_seq, p_col_seq, p_fa_seq);

    IF v_item_sia_seq IS NULL AND NOT v_trans_exist THEN
      /* check instructions to see if translation record should be created */
      v_instr_code := get_instructions(p_imp_cfg_seq, 'RES-TR', p_fa_seq);

      IF v_instr_code = 'RES-TR-01' THEN

        /* if record could not be matched to anything in translations or allowable values tables, create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq, p_fa_seq);
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not an allowable value - translation record has been created.',NULL,p_fa_seq,NULL);

      ELSE
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_item_name||', '||v_converted_item||' is not an allowable value.',NULL,p_fa_seq,NULL);
      END IF;

    ELSIF v_item_sia_seq IS NULL AND v_trans_exist AND p_orig_item != 'Actual' THEN
      /* item has been translated to an invalid storet value */
      Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_orig_item||' has been mapped to the invalid storet value: '||v_converted_item,NULL,p_fa_seq,NULL);

    END IF;

  ELSIF v_trans_exist AND v_converted_item IS NULL THEN
    /* if there is a code in the translations table but no value to translate to, write error message
       status will be changed to 'N' later in the program */
    v_item_sia_seq := NULL;
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_orig_item||' exists in translations table but has not been mapped to a STORET code.',NULL,p_fa_seq,NULL);

  END IF;
  RETURN v_item_sia_seq;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in validate allowable values.',NULL,p_fa_seq,NULL);
    RETURN NULL;

END validate_alval_item;

/******************************************************************************************************/
FUNCTION validate_storet_item(p_item_name IN VARCHAR2, p_orig_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER,
                              p_col_seq IN NUMBER, p_parent_pk IN t_storet_pk_type, p_org_id IN VARCHAR2,
                              p_line_nbr IN NUMBER, p_fa_seq IN NUMBER) RETURN t_storet_pk_type IS

  v_converted_item            VARCHAR2(80);
  v_trans_exist               BOOLEAN := TRUE;
  v_storet_pk                 t_storet_pk_type;
  v_instr_code                VARCHAR2(15);
  v_trans_ok                  BOOLEAN := TRUE;

  e_no_function               EXCEPTION;

BEGIN
  /* kms - function to look in the translations and STORET tables for items that should already exist as
     hard coded values in STORET tables */

  /* kms (5/19/01)- added p_parent_pk to parameters in order to allow validation to be dependent upon the
     presence of a valid foreign key to another STORET table, i.e. gear */

  /* look in translations table to see if it needs to swapped */
  v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, p_col_seq, v_trans_exist);

  /* kms - IF..THEN statement that calls function for the item being validated. */
  IF v_converted_item IS NOT NULL THEN

    /* call appropriate function to look in STORET table */
    IF p_item_name = 'Project ID' THEN
      v_storet_pk := get_proj_is_nbr(v_converted_item,p_org_id,p_fa_seq);
    ELSIF p_item_name = 'Lab ID' THEN
      v_storet_pk := get_lab_is_nbr(v_converted_item,p_org_id,p_fa_seq);
    ELSIF p_item_name = 'Gear ID' THEN
      v_storet_pk := get_gear_is_nbr(v_converted_item,p_fa_seq);
    ELSIF p_item_name = 'Gear Configuration ID' THEN
      v_storet_pk := get_gear_cfg_is_nbr(v_converted_item,p_parent_pk,p_org_id,p_fa_seq);
    ELSIF p_item_name = 'Sample Preservation Transport and Storage ID' THEN
      v_storet_pk := get_sample_pres_is_nbr(v_converted_item,p_org_id,p_fa_seq);
    ELSIF p_item_name = 'Sample Collection Procedure' THEN
      v_storet_pk := get_procedure_is_nbr(v_converted_item,p_org_id,p_fa_seq);
    ELSIF p_item_name = 'Station ID' THEN
      /* sf translations should only be automatically created for stations that don't exist*/
      v_storet_pk := get_station_is_nbr(v_converted_item,p_org_id,p_line_nbr,p_fa_seq);
      if v_storet_pk.org_id = 'XXX' then
        v_storet_pk.org_id := NULL;
        v_trans_ok := FALSE;
      END IF;
    ELSIF p_item_name = 'Personnel' THEN
      v_storet_pk := get_personnel_is_nbr(v_converted_item,p_orig_item,p_fa_seq);
    ELSIF p_item_name = 'Depth Units' THEN
      v_storet_pk := get_uom_is_nbr(v_converted_item,p_fa_seq);
    ELSIF p_item_name = 'Detection Limit Unit' THEN
      v_storet_pk := get_uom_is_nbr(v_converted_item,p_fa_seq);
    ELSIF p_item_name = 'Upper/Lower Depth Units' THEN
      v_storet_pk := get_uom_is_nbr(v_converted_item,p_fa_seq);
    ELSIF p_item_name = 'Characteristic Group ID' THEN
      v_storet_pk := get_group_is_nbr(v_converted_item,p_org_id,p_fa_seq);
    ELSIF p_item_name = 'Characteristic Row ID' THEN
     v_storet_pk := get_row_is_nbr(v_converted_item,p_parent_pk,p_org_id,p_fa_seq);
    ELSIF p_item_name = 'Characteristic Units'  THEN
      v_storet_pk := get_uom_is_nbr(v_converted_item,p_fa_seq);
    ELSIF p_item_name = 'Characteristic' THEN
      v_storet_pk := get_char_is_nbr(v_converted_item,p_fa_seq);
    ELSIF p_item_name = 'Field/Lab Procedure' THEN
      v_storet_pk := get_analproc_is_nbr(v_converted_item,p_org_id,p_fa_seq);
    ELSIF p_item_name = 'Lab Sample Prep Procedure' THEN
      v_storet_pk := get_labprep_is_nbr(v_converted_item,p_org_id,p_fa_seq);
    ELSE
      v_storet_pk.is_number := -1;
    END IF;

    IF v_storet_pk.is_number = -1 THEN
      RAISE e_no_function;
    END IF;

    IF v_storet_pk.is_number IS NULL AND NOT v_trans_exist THEN
      /* check instructions to see if translation record should be created */
      v_instr_code := get_instructions(p_imp_cfg_seq, 'RES-TR', p_fa_seq);

      IF v_instr_code = 'RES-TR-01' AND v_trans_ok THEN

        /* if record could not be matched to anything in translations or STORET tables,
           create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, p_col_seq, p_fa_seq);

      END IF;

    ELSIF v_storet_pk.is_number IS NULL AND v_trans_exist THEN

      /*  item has been translated to an invalid storet value */
      Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_orig_item||' has been mapped to the invalid storet value: '||v_converted_item,NULL,p_fa_seq,NULL);

    END IF;

  ELSIF v_trans_exist THEN

    /* item exists in translations table but is not mapped to anything, write error message.
       Status will be changed to 'N' later in the program. */
    v_storet_pk.is_number := NULL;
    Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': '||p_orig_item||' exists in translation table but has not been mapped to a STORET code. ',NULL,p_fa_seq,NULL);

  END IF;

  RETURN v_storet_pk;

EXCEPTION

  WHEN e_no_function THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': SIM2 does not currently handle '||p_item_name,NULL,p_fa_seq,NULL);
    v_storet_pk.is_number := NULL;
    v_storet_pk.org_id := NULL;
    RETURN v_storet_pk;

  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in validate storet item.',NULL,p_fa_seq,NULL);
    v_storet_pk.is_number := NULL;
    v_storet_pk.org_id := NULL;
    RETURN v_storet_pk;

END validate_storet_item;

/******************************************************************************************************/
FUNCTION get_activity_id(p_act_code IN VARCHAR2, p_org_id IN VARCHAR2) RETURN VARCHAR2 IS
begin
 NULL;

 /* SF 10/31/01 Removed this because it is only for special data migration purposes
 
  v_station_id              VARCHAR2(15);
  v_date                    DATE;
  v_time                    DATE;
  v_depth                   NUMBER(7,2);
  v_type_seq                NUMBER(10);
  v_medium_seq              NUMBER(10);

  v_station_is              NUMBER(8);
  v_curr_activity_id        VARCHAR2(12);
  v_year                    VARCHAR2(4);
  v_type                    VARCHAR2(1);
  v_medium                  VARCHAR2(1);
  v_sequence                VARCHAR2(5);

  CURSOR c_get_station_is_number(p_cursor_station IN VARCHAR2,p_cursor_org IN VARCHAR2) IS
    SELECT tsmstatn_is_number FROM TSMSTATN WHERE IDENTIFICATION_CD = RPAD(p_cursor_station,15)
    AND tsmstatn_org_id = RPAD(p_cursor_org,8);

  CURSOR c_get_alval_seq(p_item IN VARCHAR2, p_col_name IN VARCHAR2, p_imp_type IN VARCHAR2) IS
    SELECT SIA_SEQ FROM SIM_IMP_ALLOW_VALUES, SIM_IMP_COLS, SIM_IMP_DEFS
    WHERE SIA_SIC_SEQ = SIC_SEQ AND SIC_SID_SEQ = SID_SEQ
    AND SIA_ITEM = p_item AND sic_name = p_col_name AND SID_TYPE = p_imp_type;

  CURSOR c_check_for_activity(p_cursor_date IN DATE, p_cursor_time IN DATE,
                              p_cursor_station IN NUMBER, p_cursor_type_seq IN NUMBER,
                              p_cursor_med_seq IN NUMBER, p_cursor_org IN VARCHAR2) IS
    SELECT FA_ACTIVITY_ID FROM SIM_FIELD_ACTIVITIES WHERE TRUNC(FA_START_DATE) = TRUNC(p_cursor_date)
    AND FA_TSMSTATN_IS_NUMBER = p_cursor_station AND FA_MED_SEQ = p_cursor_med_seq
    AND FA_TYPE_SEQ = p_cursor_type_seq AND FA_TSMSTATN_ORG_ID = RPAD(p_cursor_org,8)
    AND (TRUNC(FA_START_TIME,'MI') = TRUNC(p_cursor_time,'MI')
    OR (FA_START_TIME IS NULL AND p_cursor_time IS NULL));

BEGIN
  --parse the value in the activity id column into station id, year, activity type, and medium 
  -- this should be in the format of RRMMDDHHMITMSSSSSSSS 
  BEGIN
  v_station_id := SUBSTR(p_act_code,13);
  v_year := '19'||SUBSTR(p_act_code,1,2);

  IF SUBSTR(p_act_code,7,4) = '9999' THEN
    v_time := NULL;
  ELSE
    v_time := TO_DATE(SUBSTR(p_act_code,7,4),'HH24MI');
  END IF;

  v_date := TO_DATE(SUBSTR(p_act_code,1,6),'RRMMDD');
  v_type := SUBSTR(p_act_code,11,1);
  v_medium := SUBSTR(p_act_code,12,1);
  EXCEPTION
    WHEN OTHERS THEN
      Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||'  - parsing activity code.',NULL,NULL,NULL);
  END;

  -- get station is number 
  OPEN c_get_station_is_number(v_station_id,p_org_id);
  FETCH c_get_station_is_number INTO v_station_is;
    IF c_get_station_is_number%NOTFOUND THEN
      v_station_is := NULL;
    END IF;
  CLOSE c_get_station_is_number;

  -- get type sequence 
  IF v_type = 'L' THEN
    OPEN c_get_alval_seq('Sample','Activity Type','RESULTS');
    FETCH c_get_alval_seq INTO v_type_seq;
    CLOSE c_get_alval_seq;
  ELSIF v_type = 'F' THEN
    OPEN c_get_alval_seq('Field Msr/Obs','Activity Type','RESULTS');
    FETCH c_get_alval_seq INTO v_type_seq;
    CLOSE c_get_alval_seq;
  END IF;

  -- get medium sequence 
  -- value for medium in the activity code is the second letter of the medium name when the medium starts
     with an 'S.'  So, 'O' = 'SOIL', 'E' = 'SEDIMENT.' 

  IF v_medium = 'W' THEN
    OPEN c_get_alval_seq('Water','Medium','RESULTS');
    FETCH c_get_alval_seq INTO v_medium_seq;
    CLOSE c_get_alval_seq;
  ELSIF v_medium = 'A' THEN
    OPEN c_get_alval_seq('Air','Medium','RESULTS');
    FETCH c_get_alval_seq INTO v_medium_seq;
    CLOSE c_get_alval_seq;
  ELSIF v_medium = 'E' THEN
    OPEN c_get_alval_seq('Sediment','Medium','RESULTS');
    FETCH c_get_alval_seq INTO v_medium_seq;
    CLOSE c_get_alval_seq;
  ELSIF v_medium = 'O' THEN
    OPEN c_get_alval_seq('Soil','Medium','RESULTS');
    FETCH c_get_alval_seq INTO v_medium_seq;
    CLOSE c_get_alval_seq;
  ELSIF v_medium = 'B' THEN
    OPEN c_get_alval_seq('Biological','Medium','RESULTS');
    FETCH c_get_alval_seq INTO v_medium_seq;
    CLOSE c_get_alval_seq;
  END IF;

  -- see if the result's activity already exists and create if not 
  OPEN c_check_for_activity(v_date,v_time,v_station_is,v_type_seq,v_medium_seq,RPAD(p_org_id,8));
  FETCH c_check_for_activity INTO v_curr_activity_id;
    IF c_check_for_activity%NOTFOUND THEN
      BEGIN
        -- get number sequence 
        BEGIN
          IF v_activity_count.EXISTS(v_year) THEN
            v_activity_count(v_year) := v_activity_count(v_year) + 1;
          ELSE
            v_activity_count(v_year) := 1;
          END IF;

        EXCEPTION
          WHEN OTHERS THEN
            Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq , SQLERRM||' - getting activity count.',NULL,NULL,NULL);
        END;

        v_sequence := LPAD(TO_CHAR(v_activity_count(v_year)),5,'0');

        -- create activity id
        v_curr_activity_id := v_year||'-'||v_sequence||v_type||v_medium;
      EXCEPTION
        WHEN OTHERS THEN
          Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq , SQLERRM||' - creating the activity.',NULL,NULL,NULL);
      END;
    END IF;
  CLOSE c_check_for_activity;

  -- return the activity id whether or not one was created
  RETURN v_curr_activity_id;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq , SQLERRM||' - checking for activity.',NULL,NULL,NULL);
*/
END get_activity_id;

/******************************************************************************************************/
FUNCTION check_activity_id(p_activity_id    IN VARCHAR2, p_replicate_nbr IN VARCHAR2,
                           p_station_is_nbr IN NUMBER,   p_trip_seq      IN NUMBER,
                           p_visit_seq      IN NUMBER,   p_org_id        IN VARCHAR2,
                           p_type_seq       IN NUMBER,   p_medium_seq    IN NUMBER,
                           p_status         IN OUT VARCHAR) RETURN NUMBER IS

  CURSOR c_check_activity_id(p_cursor_act_id  IN VARCHAR2, p_replicate_nbr IN VARCHAR2,
                             p_station_is_nbr IN NUMBER,   p_trip_seq      IN NUMBER,
                             p_visit_seq      IN NUMBER,   p_cursor_org    IN VARCHAR2) IS                             
    SELECT FA_SEQ, FA_TYPE_SEQ, FA_MED_SEQ FROM SIM_FIELD_ACTIVITIES
      WHERE FA_ACTIVITY_ID                  = p_cursor_act_id
      AND   NVL(FA_REPLICATE_NBR,'   ')     = NVL(p_replicate_nbr,'   ')
      AND   NVL(FA_TSMSTATN_IS_NUMBER,'-1') = NVL(p_station_is_nbr,'-1')
      AND   NVL(FA_STP_SEQ,'-1')            = NVL(p_trip_seq,'-1')
      AND   NVL(FA_SSV_SEQ,'-1')            = NVL(p_visit_seq,'-1')
      AND   RPAD(FA_TSMORGAN_ORG_ID,8)      = RPAD(p_cursor_org,8);

  v_activity_rec c_check_activity_id%ROWTYPE;

BEGIN
  /* kms - function to see if the activity already exists in SIM. */

  OPEN c_check_activity_id(p_activity_id,p_replicate_nbr,p_station_is_nbr,p_trip_seq,p_visit_seq,p_org_id);
  FETCH c_check_activity_id INTO v_activity_rec;
    IF c_check_activity_id%NOTFOUND THEN
      v_activity_rec.fa_seq := NULL;
    ELSE
      IF v_activity_rec.fa_type_seq != p_type_seq THEN
        p_status := 'N';
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': A new activity type for this activity has been found.  Only one activity type per activity is allowed.',NULL,NULL,NULL);
      ELSIF v_activity_rec.fa_med_seq != p_medium_seq THEN
        p_status := 'N';
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': A new medium for this activity has been found.  Only one medium per activity is allowed.',NULL,NULL,NULL);
      END IF;
    END IF;
  CLOSE c_check_activity_id;

  RETURN v_activity_rec.fa_seq;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in check activity id.',NULL,NULL,NULL);
    RETURN NULL;

END check_activity_id;

/******************************************************************************************************/
FUNCTION check_storet_act_id(p_activity_id IN VARCHAR2, p_replicate_nbr IN NUMBER, p_trip_is_nbr IN NUMBER,
                             p_tsmstatn_is_nbr IN NUMBER, p_org_id IN VARCHAR2, p_fa_seq IN NUMBER)
                             RETURN t_storet_pk_type IS

  v_activity_pk t_storet_pk_type;

  CURSOR c_check_storet_act_rep(p_activity_id IN VARCHAR2, p_replicate_nbr IN VARCHAR2, p_trip_is_nbr IN NUMBER,
                                p_tsmstatn_is_nbr IN NUMBER, p_org_id IN VARCHAR2) IS
    SELECT TSRFDACT_IS_NUMBER, TSRFDACT_ORG_ID FROM TSRFDACT, TSRSTVST
      WHERE ID_CODE                   = RPAD(p_activity_id,12)
      AND REPLICATE_NUMBER            = p_replicate_nbr
      AND TSRFDACT_ORG_ID             = RPAD(p_org_id,8)
      AND TSRTRIP_IS_NUMBER           = p_trip_is_nbr
      AND TSMSTATN_IS_NUMBER          = p_tsmstatn_is_nbr
      AND TSRTRIP_ORG_ID              = RPAD(p_org_id,8)
      AND TSRFDACT.TSRSTVST_IS_NUMBER = TSRSTVST.TSRSTVST_IS_NUMBER
      AND TSRFDACT.TSRSTVST_ORG_ID    = TSRSTVST.TSRSTVST_ORG_ID;

  CURSOR c_check_storet_act(p_activity_id IN VARCHAR2, p_trip_is_nbr IN NUMBER,
                            p_tsmstatn_is_nbr IN NUMBER, p_org_id IN VARCHAR2) IS
    SELECT TSRFDACT_IS_NUMBER, TSRFDACT_ORG_ID FROM TSRFDACT, TSRSTVST
      WHERE ID_CODE                   = RPAD(p_activity_id,12)
      AND REPLICATE_NUMBER            IS NULL
      AND TSRFDACT_ORG_ID             = RPAD(p_org_id,8)
      AND TSRTRIP_IS_NUMBER           = p_trip_is_nbr
      AND TSMSTATN_IS_NUMBER          = p_tsmstatn_is_nbr
      AND TSRTRIP_ORG_ID              = RPAD(p_org_id,8)
      AND TSRFDACT.TSRSTVST_IS_NUMBER = TSRSTVST.TSRSTVST_IS_NUMBER
      AND TSRFDACT.TSRSTVST_ORG_ID    = TSRSTVST.TSRSTVST_ORG_ID;

BEGIN
  /* kms - function that looks for the activity id in STORET to make sure it's not
     duplicating a pre-existing activity.  If the activity is already in STORET, SIM
     will import it but will prevent it from being exported. */

  IF p_replicate_nbr IS NOT NULL THEN
    OPEN c_check_storet_act_rep(p_activity_id,p_replicate_nbr,p_trip_is_nbr,p_tsmstatn_is_nbr,p_org_id);
    FETCH c_check_storet_act_rep INTO v_activity_pk;
      IF c_check_storet_act_rep%NOTFOUND THEN
        v_activity_pk.is_number := NULL;
        v_activity_pk.org_id := NULL;
      ELSE
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning: Activity, '||p_activity_id||' already exists on this station visit.  Activity will be prevented from being exported.',NULL,p_fa_seq,NULL);
      END IF;
    CLOSE c_check_storet_act_rep;
  ELSE
    OPEN c_check_storet_act(p_activity_id,p_trip_is_nbr,p_tsmstatn_is_nbr,p_org_id);
    FETCH c_check_storet_act INTO v_activity_pk;
      IF c_check_storet_act%NOTFOUND THEN
        v_activity_pk.is_number := NULL;
        v_activity_pk.org_id := NULL;
      ELSE
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning: Activity, '||p_activity_id||' already exists on this station visit.  Activity will be prevented from being exported.',NULL,p_fa_seq,NULL);
      END IF;
    CLOSE c_check_storet_act;
  END IF;

  RETURN v_activity_pk;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in check activity id.',NULL,p_fa_seq,NULL);
    v_activity_pk.is_number := NULL;
    v_activity_pk.org_id := NULL;
    RETURN v_activity_pk;

END check_storet_act_id;

/******************************************************************************************************/
FUNCTION get_date(p_column_name IN VARCHAR2,p_raw_date IN VARCHAR2,p_col_seq IN NUMBER,
                  p_cfg_seq IN NUMBER,p_fa_seq IN NUMBER) RETURN DATE IS

  v_start_date                  DATE;
  v_fmt                         VARCHAR2(80);

  e_no_cfg_found                EXCEPTION;

  CURSOR c_get_date_fmt(p_col_seq IN NUMBER,p_cfg_seq IN NUMBER) IS
    SELECT SIFC_FMT_CHOICE FROM SIM_IMP_CFG_DETS, SIM_IMP_FMT_CHOICES
      WHERE SICDT_SIC_SEQ = p_col_seq AND SICDT_SICC_SEQ = p_cfg_seq
      AND SICDT_SIFC_SEQ = SIFC_SEQ;

BEGIN
  /* kms - function that uses the value passed in from the text file to get a date based
     on the date format selected in the import configuration. */

  /* get format sequence from the configuration table */
  OPEN c_get_date_fmt(p_col_seq,p_cfg_seq);
  FETCH c_get_date_fmt INTO v_fmt;
    IF c_get_date_fmt%NOTFOUND THEN
      /* no format specified, use default */
      v_fmt := 'MM-DD-RRRR';
    END IF;
  CLOSE c_get_date_fmt;

  IF v_fmt IS NOT NULL THEN
    BEGIN
      v_start_date := TRUNC(TO_DATE(p_raw_date,v_fmt));
    EXCEPTION
      WHEN OTHERS THEN
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_column_name||', data does not match the format chosen in the import configuration.', NULL,p_fa_seq,NULL);
        v_start_date := NULL;
    END;
  END IF;

  RETURN v_start_date;

EXCEPTION

  WHEN e_no_cfg_found THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Import configuration for start, end, or analysis date is incomplete or does not exist.', NULL,p_fa_seq,NULL);
    RETURN NULL;

  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in get date, column '||p_column_name||' '||p_raw_date,NULL,p_fa_seq,NULL);
    RETURN NULL;

END get_date;

/******************************************************************************************************/
FUNCTION get_time(p_column_name IN VARCHAR2,p_raw_time IN VARCHAR2,p_col_seq IN NUMBER,
                  p_cfg_seq IN NUMBER,p_fa_seq IN NUMBER) RETURN DATE IS

  v_start_time                  DATE;
  v_fmt                         VARCHAR2(80);

  e_no_cfg_found                EXCEPTION;

  CURSOR c_get_date_fmt(p_col_seq IN NUMBER,p_cfg_seq IN NUMBER) IS
    SELECT SIFC_FMT_CHOICE FROM SIM_IMP_CFG_DETS, SIM_IMP_FMT_CHOICES
      WHERE SICDT_SIC_SEQ = p_col_seq AND SICDT_SICC_SEQ = p_cfg_seq
      AND SICDT_SIFC_SEQ = SIFC_SEQ;

BEGIN
  /* kms - function that converts the value in the text file to a time based on the the format
     selected in the import configuration */

  /* get format sequence from the configuration table */
  OPEN c_get_date_fmt(p_col_seq,p_cfg_seq);
  FETCH c_get_date_fmt INTO v_fmt;
    IF c_get_date_fmt%NOTFOUND THEN
      /* no format specified, use default */
      v_fmt := 'HH24:MI:SS';
    END IF;
  CLOSE c_get_date_fmt;

  IF v_fmt IS NOT NULL THEN
    BEGIN
      v_start_time := TO_DATE(SUBSTR(p_raw_time,1,8),v_fmt);
    EXCEPTION
      WHEN OTHERS THEN
        v_start_time := NULL;
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_column_name||', data does not match the format chosen in the import configuration.', NULL,p_fa_seq,NULL);
    END;
  END IF;

  RETURN v_start_time;

EXCEPTION

  WHEN e_no_cfg_found THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Import configuration for start or end time is incomplete or does not exist.', NULL,p_fa_seq,NULL);
    RETURN NULL;

  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in get time '||p_raw_time||', column '||p_column_name,NULL,p_fa_seq,NULL);
    RETURN NULL;

END get_time;

/******************************************************************************************************/
FUNCTION check_gear_type(p_proc_is_nbr IN NUMBER, p_gear_is_nbr IN NUMBER, p_org_id IN VARCHAR2, p_fa_seq IN NUMBER)
  RETURN NUMBER IS

  v_gear_gear_type       VARCHAR2(25);
  v_proc_gear_type       VARCHAR2(25);
  v_gear_type_ok         NUMBER(1);

  e_proc_not_found       EXCEPTION;

  CURSOR c_get_gear_gear_type (p_gear_is_nbr IN NUMBER) IS
    SELECT TYPE_NAME FROM TSRFLDGR WHERE TSRFLDGR_IS_NUMBER = p_gear_is_nbr;

  CURSOR c_get_proc_gear_type (p_proc_is_nbr IN NUMBER, p_org_id IN VARCHAR2) IS
    SELECT FLD_GEAR_TYPE_NAME FROM TSRFLDPR WHERE TSRFLDPR_IS_NUMBER = p_proc_is_nbr
      AND TSRFLDPR_ORG_ID = RPAD(p_org_id,8);

BEGIN
  /* kms - function that makes sure the gear and field procedure selected are compatible -
     that they have the same gear type */

  /* kms (5/18/01) - Changed this so that it only checks the gear types if the collection has
     been assigned a gear type and gear has been given.  This function will be called
     whenever sample collection procedure is present, whether or not gear is also there. */

  /* get gear type from field procedure */
  OPEN c_get_proc_gear_type(p_proc_is_nbr, p_org_id);
  FETCH c_get_proc_gear_type INTO v_proc_gear_type;
    IF c_get_proc_gear_type%NOTFOUND THEN
      RAISE e_proc_not_found;
    END IF;
  CLOSE c_get_proc_gear_type;

  IF p_gear_is_nbr IS NOT NULL AND (RPAD(v_proc_gear_type,25) != RPAD(' ',25) AND v_proc_gear_type IS NOT NULL) THEN
    /* kms - if gear has been given, make sure gear types match */
    /* get gear type from gear */
    OPEN c_get_gear_gear_type(p_gear_is_nbr);
    FETCH c_get_gear_gear_type INTO v_gear_gear_type;
      IF c_get_gear_gear_type%NOTFOUND THEN
        v_gear_gear_type := NULL;
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Error finding gear type in TSRFLDGR, is number:'||p_gear_is_nbr,NULL,p_fa_seq,NULL);
      END IF;
    CLOSE c_get_gear_gear_type;

    /* compare gear types and set flag to true if ok, else set to false */
    IF v_gear_gear_type = v_proc_gear_type THEN
      v_gear_type_ok := 0;
    ELSE
      v_gear_type_ok := 1;
    END IF;

  ELSIF p_gear_is_nbr IS NULL AND (RPAD(v_proc_gear_type,25) != RPAD(' ',25) AND v_proc_gear_type IS NOT NULL) THEN
    /* kms - if gear has not been given and the procedure has a gear type, set flag to invalidate
       the activity and write an error message in import results procedure. */
     v_gear_type_ok := 2;

  END IF;

  RETURN v_gear_type_ok;

EXCEPTION
  WHEN e_proc_not_found THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Error finding gear type in TSRFLDPR, is number:'||p_proc_is_nbr,NULL,p_fa_seq,NULL);
    RETURN NULL;

  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in check gear type.',NULL,p_fa_seq,NULL);
    RETURN NULL;

END check_gear_type;

/******************************************************************************************************/
FUNCTION check_trip_dates (p_trip_seq IN NUMBER, p_activity_date IN DATE, p_org_id IN VARCHAR2,
                           p_fa_seq IN NUMBER) RETURN NUMBER IS

  v_start_date           DATE;
  v_end_date             DATE;
  v_after                NUMBER(10);
  v_before               NUMBER(10);
  v_return_value         NUMBER(1) := 1;

  CURSOR c_trip_dates (p_trip_seq IN NUMBER) IS
      SELECT STP_START_DATE, STP_END_DATE FROM SIM_TRIPS WHERE STP_SEQ = p_trip_seq;

BEGIN
  /* kms - function that makes sure the activity occured between the start and end dates of the trip.
     Called by main import procedure - import_results. */

  OPEN c_trip_dates(p_trip_seq);
  FETCH c_trip_dates INTO v_start_date, v_end_date;
    IF c_trip_dates%NOTFOUND THEN
      CLOSE c_trip_dates;
      RETURN 0;
    END IF;
  CLOSE c_trip_dates;

  /* added for STORET interface bug where some trip years have been entered as 0001 */
  IF TO_CHAR(p_activity_date,'RRRR') LIKE '20__' THEN
    IF TO_CHAR(v_start_date,'RRRR') LIKE '00__' THEN
      v_start_date := TO_DATE(TO_CHAR(v_start_date,'MM-DD')||'-2'||SUBSTR(TO_CHAR(v_start_date,'RRRR'),2,3),'MM-DD-RRRR');
    END IF;

    IF TO_CHAR(v_end_date,'RRRR') LIKE '00__' THEN
      v_end_date := TO_DATE(TO_CHAR(v_end_date,'MM-DD')||'-2'||SUBSTR(TO_CHAR(v_end_date,'RRRR'),2,3),'MM-DD-RRRR');
    END IF;
  END IF;

  /* make sure activity date is within trip date range */
  v_after := v_start_date - p_activity_date;  --will be negative if activity after start date

  /* if activity came after trip start date, make sure it came before trip end date */
  IF v_after <= 0 THEN

    IF TO_CHAR(v_end_date,'RRRR') IS NOT NULL THEN
      v_before := 0; -- bka  p_activity_date - v_end_date;  --will be negative if activity before end date

      IF v_before > 0 THEN
        /* activity occured after the trip ended, prevent export, write error message */
        v_return_value := 0;
        Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, 'Line '||v_line_cnt||': Activity start date is after the end date for the indicated trip. The trip will be left off of the activity.',NULL,p_fa_seq,NULL);
      END IF;
    END IF;
  ELSE
    /* activity occured before the trip began, prevent export, write error message */
    v_return_value := 0;
    Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, 'Line '||v_line_cnt||': Activity start date is before the start date for the indicated trip. The trip will be left off of the activity.',NULL,p_fa_seq,NULL);

  END IF;

  RETURN v_return_value;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in check trip dates.',NULL,p_fa_seq,NULL);
    RETURN 0;

END check_trip_dates;

/******************************************************************************************************/
FUNCTION check_visit_dates (p_visit_seq IN NUMBER, p_activity_date IN DATE, p_org_id IN VARCHAR2,
                            p_fa_seq IN NUMBER) RETURN NUMBER IS

  v_start_date           DATE;
  v_end_date             DATE;
  v_after                NUMBER(10);
  v_before               NUMBER(10);
  v_return_value         NUMBER(1) := 1;

  CURSOR c_visit_dates (p_visit_seq IN NUMBER) IS
      SELECT SSV_START_DATE, SSV_END_DATE FROM SIM_STATION_VISITS WHERE SSV_SEQ = p_visit_seq;

BEGIN
  /* kms - function that makes sure the activity occured between the start and end dates of the visit.
     Called by main import procedure - import_results. */

  OPEN c_visit_dates(p_visit_seq);
  FETCH c_visit_dates INTO v_start_date, v_end_date;
    IF c_visit_dates%NOTFOUND THEN
      CLOSE c_visit_dates;
      RETURN 0;
    END IF;
  CLOSE c_visit_dates;


 -- IF v_end_date IS NULL THEN    (sef 9/25/01 that's okay!)
--    v_end_date := p_activity_date;
 -- END IF;

  /* added for STORET interface bug where some years have been entered as 0001 */
  IF TO_CHAR(p_activity_date,'RRRR') LIKE '20__' THEN
    IF TO_CHAR(v_start_date,'RRRR') LIKE '00__' THEN
      v_start_date := TO_DATE(TO_CHAR(v_start_date,'MM-DD')||'-2'||SUBSTR(TO_CHAR(v_start_date,'RRRR'),2,3),'MM-DD-RRRR');
    END IF;

   -- IF TO_CHAR(v_end_date,'RRRR') LIKE '00__' THEN
   --   v_end_date := TO_DATE(TO_CHAR(v_end_date,'MM-DD')||'-2'||SUBSTR(TO_CHAR(v_end_date,'RRRR'),2,3),'MM-DD-RRRR');
   -- END IF;
  END IF;

  /* make sure activity date is within visit date range */
  v_after := v_start_date - p_activity_date;  --will be negative if activity after start date

  /* if activity came after visit start date, make sure it came before visit end date
  sef 9/25/01 Changing this because we don't want to check end dates for now
  */
  --IF v_after <= 0 THEN

    --IF TO_CHAR(v_end_date,'RRRR') IS NOT NULL THEN
    --  v_before := p_activity_date - v_end_date;  --will be negative if activity before end date

    --  IF v_before > 0 THEN
        /* activity occured after the visit ended, prevent export, write error message */
    --    v_return_value := 0;
    --    Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, 'Line '||v_line_cnt||': Activity start date is after the end date for the indicated station visit. The station visi will be left off of the activity.',NULL,p_fa_seq,NULL);
    --  END IF;
    --END IF;
--  ELSE

    /* activity occured before the visit began, prevent export, write error message */
  IF v_after > 0 THEN
    v_return_value := 0;
    Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, 'Line '||v_line_cnt||': Activity start date is before the start date for the indicated station visit. The station visit will be left off of the activity.',NULL,p_fa_seq,NULL);
  END IF;

  RETURN v_return_value;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in check visit dates.',NULL,p_fa_seq,NULL);
    RETURN 0;

END check_visit_dates;

/******************************************************************************************************/
FUNCTION get_trip_id(p_instr_code IN VARCHAR2, p_start_date IN DATE, p_trip_start_date OUT DATE,
                     p_trip_end_date OUT DATE, p_org_id IN VARCHAR2, p_fa_seq IN NUMBER) RETURN VARCHAR2 IS

  v_trip_id              VARCHAR2(15);

  CURSOR c_get_trip(p_trip_id IN VARCHAR2, p_org_id IN VARCHAR2) IS
    SELECT TSRTRIP_IS_NUMBER, TSRTRIP_ORG_ID FROM TSRTRIP WHERE ID_CODE = RPAD(p_trip_id,15)
      AND TSRTRIP_ORG_ID = RPAD(p_org_id,8);

BEGIN
  /* kms - function that will get the trip primary key based on instructions - id will be based either
     on year, month and year, date, or on what's in the text file.  Will create trip only if the id
     cannot be found in STORET. */

  /* one trip per year */
  IF p_instr_code = 'RES-TRIP-01' THEN
    v_trip_id := TO_CHAR(p_start_date,'RRRR');

    /* trip starts on the first day of the year */
    p_trip_start_date := TO_DATE('1-1-'||TO_CHAR(TRUNC(p_start_date,'RRRR'),'RRRR'),'MM-DD-RRRR');

    /* trip ends on the last day of the year */
    p_trip_end_date := TO_DATE('12-31-'||TO_CHAR(TRUNC(p_start_date,'RRRR'),'RRRR'),'MM-DD-RRRR');

  /* one trip per month */
  ELSIF p_instr_code = 'RES-TRIP-02' THEN
    v_trip_id := TO_CHAR(p_start_date,'MMRRRR');

    /* trip starts on the first day of the month */
    p_trip_start_date := TRUNC(p_start_date,'MON');

    /* trip ends on the last day of the month */
    p_trip_end_date := LAST_DAY(p_start_date);

  /* one trip per day */
  ELSIF p_instr_code = 'RES-TRIP-03' THEN
    v_trip_id := TO_CHAR(p_start_date,'MMDDRRRR');

    /* trip starts the same day */
    p_trip_start_date := p_start_date;

    /* trip ends same day */
    p_trip_end_date := p_start_date;

  END IF;

  RETURN v_trip_id;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq , 'Line '||v_line_cnt||': '||SQLERRM||' - in get trip id.',NULL,p_fa_seq,NULL);
    p_trip_start_date := NULL;
    p_trip_end_date := NULL;
    v_trip_id := NULL;
    RETURN v_trip_id;

END get_trip_id;

/******************************************************************************************************/
FUNCTION create_sim_trip(p_trip_rec IN SIM_TRIPS%ROWTYPE, p_fa_seq IN NUMBER)
  RETURN NUMBER IS

BEGIN

  /* create sim_trips using values in trip record */
  INSERT INTO SIM_TRIPS (
    STP_SEQ,
    STP_TRIP_ID,
    STP_START_DATE,
    STP_START_TIME,
    STP_END_DATE,
    STP_END_TIME,
    STP_TSRTRIP_IS_NUMBER,
    STP_TSRTRIP_ORG_ID,
    STP_TSMORGAN_IS_NUMBER,
    STP_TSMORGAN_ORG_ID,
    STP_CREATED_BY_SIM
    )
  VALUES (
    p_trip_rec.STP_SEQ,                      /* STP_SEQ */
    p_trip_rec.STP_TRIP_ID,                  /* STP_TRIP_ID */
    p_trip_rec.STP_START_DATE,               /* STP_START_DATE */
    p_trip_rec.STP_START_TIME,               /* STP_START_TIME */
    p_trip_rec.STP_END_DATE,                 /* STP_END_DATE */
    p_trip_rec.STP_END_TIME,                 /* STP_END_TIME */
    p_trip_rec.STP_TSRTRIP_IS_NUMBER,        /* STP_TSRTRIP_IS_NUMBER */
    p_trip_rec.STP_TSRTRIP_ORG_ID,           /* STP_TSRTRIP_ORG_ID */
    p_trip_rec.STP_TSMORGAN_IS_NUMBER,       /* STP_TSMORGAN_IS_NUMBER */
    p_trip_rec.STP_TSMORGAN_ORG_ID,          /* STP_TSMORGAN_ORG_ID */
    p_trip_rec.STP_CREATED_BY_SIM            /* STP_CREATED_BY_SIM */
    );

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq , 'Line '||v_line_cnt||': '||SQLERRM||' - creating SIM trip.',NULL,p_fa_seq,NULL);
    RETURN NULL;

END create_sim_trip;

/******************************************************************************************************/
PROCEDURE get_or_add_trip_rec(p_trip_rec IN OUT SIM_TRIPS%ROWTYPE, p_org_id IN VARCHAR2, p_fa_seq IN NUMBER, 
                              p_trip_start_date IN DATE) IS

  v_trip_seq                 NUMBER(10);
  v_trip_start_date          DATE;
  v_trip_end_date            DATE;
  v_success                  NUMBER(1);
  
  e_no_start_date            EXCEPTION;

  CURSOR c_get_trip_seq(p_trip_id IN VARCHAR2, p_org_id IN VARCHAR2) IS
    SELECT STP_SEQ, STP_TSRTRIP_IS_NUMBER, STP_TSRTRIP_ORG_ID FROM SIM_TRIPS
      WHERE STP_TRIP_ID = p_trip_id AND STP_TSMORGAN_ORG_ID = RPAD(p_org_id,8);

  CURSOR c_get_trip_key(p_trip_id IN VARCHAR2, p_org_id IN VARCHAR2) IS
    SELECT TSRTRIP_IS_NUMBER, TSRTRIP_ORG_ID, START_DATE, START_TIME FROM TSRTRIP
      WHERE ID_CODE = RPAD(p_trip_id,15) AND TSRTRIP_ORG_ID = RPAD(p_org_id,8);

  CURSOR c_get_next_trip_seq IS SELECT SIM_TRIPS_SEQ.NEXTVAL FROM DUAL;

BEGIN
  /* kms (6/4/01)- function that will look for the trip id in the SIM table.  If it's not
     there this it will then look in STORET to see if the trip exists there and, if so, it
     will retrieve the primary key from the trips table to be used when the trip is added
     to SIM.  This function will add the trip to SIM if it is not already there and return
     the sequence to the trip record to be used as a reference when station visits are
     processed. */

  /* kms - check SIM */
  OPEN c_get_trip_seq(p_trip_rec.STP_TRIP_ID,p_org_id);
  FETCH c_get_trip_seq INTO v_trip_seq, p_trip_rec.STP_TSRTRIP_IS_NUMBER, p_trip_rec.STP_TSRTRIP_ORG_ID;
    IF c_get_trip_seq%NOTFOUND THEN

      v_trip_seq := NULL;
      p_trip_rec.STP_TSMORGAN_IS_NUMBER := v_org_is_nbr;
      p_trip_rec.STP_TSMORGAN_ORG_ID := p_org_id;

      /* kms - trip is not in SIM, see if it is in STORET */
      OPEN c_get_trip_key(p_trip_rec.STP_TRIP_ID, p_org_id);
      FETCH c_get_trip_key INTO p_trip_rec.STP_TSRTRIP_IS_NUMBER,p_trip_rec.STP_TSRTRIP_ORG_ID,v_trip_start_date,v_trip_end_date;
        IF c_get_trip_key%NOTFOUND THEN
          /* kms - trip is not in STORET, set start date to default if it is null and flag accordingly */
          p_trip_rec.STP_CREATED_BY_SIM := 1;
          /* bka - If the trip start date is not found, use the one from the data file. */
          IF p_trip_rec.STP_START_DATE IS NULL THEN
            IF p_trip_start_date IS NOT NULL THEN
              p_trip_rec.STP_START_DATE := p_trip_start_date;
            ELSE
              RAISE e_no_start_date;
            END IF;
          END IF;
        ELSE
          /* kms - trip is in STORET, set flag to show it wasn't created by SIM */
          p_trip_rec.STP_CREATED_BY_SIM := 0;
          p_trip_rec.STP_START_DATE := v_trip_start_date;
          p_trip_rec.STP_END_DATE := v_trip_end_date;
        END IF;
      CLOSE c_get_trip_key;

    END IF;
  CLOSE c_get_trip_seq;
  IF v_trip_seq IS NULL THEN
    /* kms - get sequence for new trip */
    OPEN c_get_next_trip_seq;
    FETCH c_get_next_trip_seq INTO p_trip_rec.STP_SEQ;
    CLOSE c_get_next_trip_seq;

    /* kms - create trip in SIM */
    v_success := create_sim_trip(p_trip_rec,p_fa_seq);
  ELSE
    p_trip_rec.STP_SEQ := v_trip_seq;
  END IF;

EXCEPTION
  WHEN e_no_start_date THEN
    Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq , 'Line '||v_line_cnt||': Trip Start Date is not available, Trip cannot be found or created.',
                                NULL,p_fa_seq,NULL);
    p_trip_rec.STP_TSRTRIP_ORG_ID := NULL;
    p_trip_rec.STP_TSRTRIP_IS_NUMBER := NULL;
    p_trip_rec.STP_SEQ := NULL;
  WHEN OTHERS THEN
    Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq , 'Line '||v_line_cnt||': '||SQLERRM||' - in get or add trip rec.',NULL,p_fa_seq,NULL);
    p_trip_rec.STP_TSRTRIP_ORG_ID := NULL;
    p_trip_rec.STP_TSRTRIP_IS_NUMBER := NULL;
    p_trip_rec.STP_SEQ := NULL;

END get_or_add_trip_rec;

/******************************************************************************************************/
FUNCTION get_next_project_is_nbr (p_org_id IN VARCHAR2, p_fa_seq IN NUMBER) RETURN NUMBER IS

BEGIN
NULL;

/* Don't need this any more SF-10/5/01
  -- kms - gets the next project is number.  If this has not been done yet, uses the value in STORET1's
     --TSMEISN table, otherwise, it adds one to the last number it used. 

   IF v_cur_proj_is_nbr IS NULL THEN
      v_cur_proj_is_nbr := Sim2_Export_Pkg.get_eisn_number('TSMPROJ',p_org_id);
   ELSE
      v_cur_proj_is_nbr := v_cur_proj_is_nbr + 1;
   END IF;

   RETURN v_cur_proj_is_nbr;

EXCEPTION
   WHEN OTHERS THEN
     Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in get next project is nbr.', NULL, p_fa_seq, NULL);
     RETURN NULL;
*/
END get_next_project_is_nbr;

/******************************************************************************************************/
FUNCTION add_project(p_project_id IN VARCHAR2,p_org_id IN VARCHAR2, p_fa_seq IN NUMBER) RETURN NUMBER IS
BEGIN
NULL;
/* We don't add projects any more. SF-10/5/01

  v_new_proj_is_nbr   NUMBER(8):=0;
  v_start_date        DATE;
  v_comment           VARCHAR2(15);

BEGIN
  -- kms - function that creates a project if its processed id could not be found in STORET. 

  -- get new is number 
  v_new_proj_is_nbr := get_next_project_is_nbr(p_org_id, p_fa_seq);

  v_start_date := TO_DATE('01-JAN-1000','DD_MON_YYYY');
  v_comment := ('SIM: '||v_import_log_seq);

 BEGIN

  -- add project to storet 
  INSERT INTO TSMPROJ(
    TSMPROJ_IS_NUMBER,
    TSMPROJ_ORG_ID,
    IDENTIFICATION_CD,
    NAME,
    START_DATE,
    PLANNED_DURATION,
    D_USERID_CODE,
    D_LAST_UPDT_TS,
    TSMORGAN_IS_NUMBER
    )
  VALUES(
    v_new_proj_is_nbr,      -- TSMPROJ_IS_NUMBER 
    p_org_id,		            -- TSMPROJ_ORG_ID 
    RPAD(p_project_id,8),   -- IDENTIFICATION_CD 
    p_project_id,           -- NAME 
    v_start_date,           -- START_DATE 
    v_comment,              -- PLANNED_DURATION 
    USER,				            -- D_USERID_CODE 
    SYSDATE,		            -- D_LAST_UPDT_TS 
    v_org_is_nbr 			      -- TSMORGAN_IS_NUMBER 
    );

  EXCEPTION
    WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq , 'Line '||v_line_cnt||': '||SQLERRM||' - inserting project record: '||p_project_id,NULL,p_fa_seq,NULL);
  END;

  RETURN v_new_proj_is_nbr;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq , 'Line '||v_line_cnt||': '||SQLERRM||' - creating project: '||p_project_id,NULL,p_fa_seq,NULL);
    RETURN NULL;
*/
END add_project;

/******************************************************************************************************/
FUNCTION create_sim_station_visit(p_visit_rec IN SIM_STATION_VISITS%ROWTYPE)
  RETURN NUMBER IS

BEGIN
  /* create sim_station_visits using values in visit record */
  INSERT INTO SIM_STATION_VISITS (
    SSV_SEQ,
    SSV_STP_SEQ,
    SSV_ID,
    SSV_START_DATE,
    SSV_START_TIME,
    SSV_END_DATE,
    SSV_END_TIME,
    SSV_TSMSTATN_IS_NUMBER,
    SSV_TSMSTATN_ORG_ID,
    SSV_TSRSTVST_IS_NUMBER,
    SSV_TSRSTVST_ORG_ID,
    SSV_TSMORGAN_IS_NUMBER,
    SSV_TSMORGAN_ORG_ID,
    SSV_CREATED_BY_SIM
    )
  VALUES (
    p_visit_rec.SSV_SEQ,                       /* SSV_SEQ */
    p_visit_rec.SSV_STP_SEQ,                   /* SSV_STP_SEQ */
    p_visit_rec.SSV_ID,                        /* SSV_ID */
    p_visit_rec.SSV_START_DATE,                /* SSV_START_DATE */
    p_visit_rec.SSV_START_TIME,                /* SSV_START_TIME */
    p_visit_rec.SSV_END_DATE,                  /* SSV_END_DATE */
    p_visit_rec.SSV_END_TIME,                  /* SSV_END_TIME */
    p_visit_rec.SSV_TSMSTATN_IS_NUMBER,        /* SSV_TSMSTATN_IS_NUMBER */
    p_visit_rec.SSV_TSMSTATN_ORG_ID,           /* SSV_TSMSTATN_ORG_ID */
    p_visit_rec.SSV_TSRSTVST_IS_NUMBER,        /* SSV_TSRSTVST_IS_NUMBER */
    p_visit_rec.SSV_TSRSTVST_ORG_ID,           /* SSV_TSRSTVST_ORG_ID */
    p_visit_rec.SSV_TSMORGAN_IS_NUMBER,        /* SSV_TSMORGAN_IS_NUMBER */
    p_visit_rec.SSV_TSMORGAN_ORG_ID,           /* SSV_TSMORGAN_ORG_ID */
    p_visit_rec.SSV_CREATED_BY_SIM             /* SSV_CREATED_BY_SIM */
    );

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq , 'Line '||v_line_cnt||': '||SQLERRM||' - creating sim station_visit.',NULL,NULL,NULL);
    RETURN NULL;

END create_sim_station_visit;

/******************************************************************************************************/
PROCEDURE load_station_visits(p_trip_seq IN NUMBER, p_trip_key IN t_storet_pk_type, p_org_id IN VARCHAR2,
                              p_stvst_id_table IN OUT t_stvst_id_type, p_fa_seq IN NUMBER) IS

  v_id_nbr                    NUMBER(3);
  v_max_nbr                   NUMBER(3) := 0;
  v_trip_sta_idx              NUMBER(19);
  v_visit_seq                 NUMBER(10);
  v_visit_rec                 SIM_STATION_VISITS%ROWTYPE;
  v_last_statn_is_nbr         NUMBER(8);
  v_success                   NUMBER(1);

  CURSOR c_get_trip_station_visits(p_trip_is_nbr IN NUMBER, p_trip_org_id IN VARCHAR2) IS
    SELECT ID_NUMBER vst_id, ARRIVAL_DATE, ARRIVAL_TIME, DEPARTURE_DATE, DEPARTURE_TIME,
      TSRSTVST_IS_NUMBER is_nbr, TSRSTVST_ORG_ID org_id, TSMSTATN_IS_NUMBER statn_is_nbr,
      TSMSTATN_ORG_ID statn_org_id FROM TSRSTVST
      WHERE TSRTRIP_IS_NUMBER = p_trip_is_nbr AND TSRTRIP_ORG_ID = RPAD(p_trip_org_id,8)
      ORDER BY TSMSTATN_IS_NUMBER;

  CURSOR c_check_visit(p_visit_is_nbr IN NUMBER, p_visit_org_id IN VARCHAR2) IS
    SELECT SSV_SEQ FROM SIM_STATION_VISITS WHERE SSV_TSRSTVST_IS_NUMBER = p_visit_is_nbr
      AND SSV_TSRSTVST_ORG_ID = RPAD(p_visit_org_id,8);

  CURSOR c_get_next_visit_seq IS
    SELECT SIM_STATION_VISITS_SEQ.NEXTVAL FROM DUAL;

BEGIN
  /* kms (6/8/01) - get all the station visits that exist on this trip in STORET and load
     them into the SIM table if they are not already there.  Update visits that are already
     in SIM with what's in STORET. */

  /* kms - loop through all station visits on the current trip */
  FOR v_visit IN c_get_trip_station_visits(p_trip_key.is_number, p_trip_key.org_id) LOOP

     /* kms - see if station visit is already in SIM_STATION_VISITS and add if not. */
    OPEN c_check_visit(v_visit.is_nbr, v_visit.org_id);
    FETCH c_check_visit INTO v_visit_seq;
      IF c_check_visit%NOTFOUND THEN

        /* get sequence for new record */
        OPEN c_get_next_visit_seq;
        FETCH c_get_next_visit_seq INTO v_visit_seq;
        CLOSE c_get_next_visit_seq;

        /* add station visit to sim */
        v_visit_rec.SSV_SEQ := v_visit_seq;
        v_visit_rec.SSV_STP_SEQ := p_trip_seq;
        v_visit_rec.SSV_ID := v_visit.vst_id;
        v_visit_rec.SSV_START_DATE := v_visit.arrival_date;
        v_visit_rec.SSV_START_TIME := v_visit.arrival_time;
        v_visit_rec.SSV_END_DATE := v_visit.departure_date;
        v_visit_rec.SSV_END_TIME := v_visit.departure_time;
        v_visit_rec.SSV_TSRSTVST_IS_NUMBER := v_visit.is_nbr;
        v_visit_rec.SSV_TSRSTVST_ORG_ID := v_visit.org_id;
        v_visit_rec.SSV_TSMSTATN_IS_NUMBER := v_visit.statn_is_nbr;
        v_visit_rec.SSV_TSMSTATN_ORG_ID := v_visit.statn_org_id;
        v_visit_rec.SSV_TSMORGAN_IS_NUMBER := v_org_is_nbr;
        v_visit_rec.SSV_TSMORGAN_ORG_ID := p_org_id;
        v_visit_rec.SSV_CREATED_BY_SIM := 0;

        v_success := create_sim_station_visit(v_visit_rec);
      ELSIF c_check_visit%FOUND THEN

        /* kms - update station visit in SIM with values in STORET. */
        UPDATE SIM_STATION_VISITS
          SET SSV_ID = v_visit.vst_id,
            SSV_START_DATE = v_visit.arrival_date,
            SSV_START_TIME = v_visit.arrival_time,
            SSV_END_DATE = v_visit.departure_date,
            SSV_END_TIME = v_visit.departure_time
          WHERE SSV_SEQ = v_visit_seq;

      END IF;
    CLOSE c_check_visit;

  END LOOP;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq , 'Line '||v_line_cnt||': '||SQLERRM||' - in load_station_visits.',NULL,p_fa_seq,NULL);

END load_station_visits;

/******************************************************************************************************/
FUNCTION get_visit_start_date (p_visit_key IN t_storet_pk_type, p_trip_seq IN NUMBER,
                               p_fa_seq IN NUMBER) RETURN DATE IS

  v_start_date                    DATE;
  v_success                       NUMBER(1);

  CURSOR c_get_date_from_STORET(p_tsrstvst_is_nbr IN NUMBER, p_tsrstvst_org_id IN VARCHAR2) IS
    SELECT ARRIVAL_DATE FROM TSRSTVST WHERE TSRSTVST_IS_NUMBER = p_tsrstvst_is_nbr
      AND TSRSTVST_ORG_ID = RPAD(p_tsrstvst_org_id,8);

  CURSOR c_get_date_from_trip(p_trip_seq IN NUMBER) IS
    SELECT STP_START_DATE FROM SIM_TRIPS WHERE STP_SEQ = p_trip_seq;

BEGIN
  /* kms (5/30/01)- function that gets the station visit start date from the station
     visit if it exists in STORET or from the trip. */
  /* kms (6/02/01 - changed this so it only gets it from the trip.  Since station
     visits are now loaded into SIM when a trip is first encountered by the import, we
     know that all of the station visits that exist in STORET are also in SIM.  The
     start date, at this point is only needed if the visit does not exist in SIM (and
     is therefore not in STORET). */

    /* kms - station visit does not exist yet, get date from trip in SIM table.
       This will be replaced later if the station visit start date has been
       included. */
    OPEN c_get_date_from_trip(p_trip_seq);
    FETCH c_get_date_from_trip INTO v_start_date;
      IF c_get_date_from_trip%NOTFOUND THEN
        v_start_date := NULL;
        Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, 'Line '||v_line_cnt||': Error retrieving start date from SIM_TRIPS.',NULL,p_fa_seq,NULL);
      END IF;
    CLOSE c_get_date_from_trip;

  RETURN v_start_date;

EXCEPTION
   WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in get visit start date.',NULL,p_fa_seq,NULL);
      RETURN NULL;

END get_visit_start_date;

/******************************************************************************************************/
FUNCTION get_visit_seq (p_visit_key OUT t_storet_pk_type, p_station_key IN t_storet_pk_type,
                        p_visit_id IN VARCHAR2, p_trip_seq IN NUMBER, p_fa_seq IN NUMBER) RETURN NUMBER IS

  v_visit_seq                     NUMBER(10);
  v_visit_rec                     SIM_STATION_VISITS%ROWTYPE;
  v_success                       NUMBER(1);
  v_visit_id                      NUMBER;

  CURSOR c_find_station_visit_c(p_visit_nbr   IN VARCHAR2, p_stat_is_nbr IN NUMBER,
                                p_stat_org_id IN VARCHAR2, p_trip_seq    IN NUMBER) IS
    SELECT SSV_SEQ, SSV_TSRSTVST_IS_NUMBER, SSV_TSRSTVST_ORG_ID
    FROM SIM_STATION_VISITS
    WHERE SSV_ID = p_visit_nbr
    AND SSV_TSMSTATN_IS_NUMBER = p_stat_is_nbr
    AND SSV_TSMSTATN_ORG_ID = RPAD(p_stat_org_id,8)
    AND SSV_STP_SEQ = p_trip_seq;
    
  CURSOR c_find_station_visit_n(p_visit_nbr   IN NUMBER,   p_stat_is_nbr IN NUMBER,
                                p_stat_org_id IN VARCHAR2, p_trip_seq    IN NUMBER) IS
    SELECT SSV_SEQ, SSV_TSRSTVST_IS_NUMBER, SSV_TSRSTVST_ORG_ID
    FROM SIM_STATION_VISITS
    WHERE SSV_ID = p_visit_nbr
    AND SSV_TSMSTATN_IS_NUMBER = p_stat_is_nbr
    AND SSV_TSMSTATN_ORG_ID = RPAD(p_stat_org_id,8)
    AND SSV_STP_SEQ = p_trip_seq;

  CURSOR c_get_next_visit_seq IS
    SELECT SIM_STATION_VISITS_SEQ.NEXTVAL FROM DUAL;

BEGIN
  /* kms (6/2/01)- function that sees if station visit already exists in sim_station_visits
     and returns its sequence if it does.  Called only when instructions are to read the
     station visit id from the data file. */
     
  /* kms - find the station visit with this id that for the station on the current activity */
  
  BEGIN -- bka - try to convert p_visit_id to a number
    v_visit_id := TO_NUMBER(p_visit_id);  
    OPEN c_find_station_visit_n(v_visit_id,p_station_key.is_number,
                                p_station_key.org_id, p_trip_seq);
    FETCH c_find_station_visit_n INTO v_visit_seq,p_visit_key.is_number,p_visit_key.org_id;
      IF c_find_station_visit_n%NOTFOUND THEN
        v_visit_seq := NULL;
      END IF;
    CLOSE c_find_station_visit_n; 
  EXCEPTION WHEN OTHERS THEN -- if p_visit_id is not a number pass it as varchar2
    OPEN c_find_station_visit_c(p_visit_id,p_station_key.is_number,
                                p_station_key.org_id, p_trip_seq);
    FETCH c_find_station_visit_c INTO v_visit_seq,p_visit_key.is_number,p_visit_key.org_id;
      IF c_find_station_visit_c%NOTFOUND THEN
        v_visit_seq := NULL;
      END IF;
    CLOSE c_find_station_visit_c;
  END; 

  RETURN v_visit_seq;

EXCEPTION
  WHEN OTHERS THEN
     Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in get visit seq.',NULL,p_fa_seq,NULL);
     v_visit_seq := NULL;
     RETURN v_visit_seq;

END get_visit_seq;

/******************************************************************************************************/
FUNCTION check_visit_start_date(p_statn_is_number IN NUMBER, p_visit_id IN NUMBER, 
                                p_visit_start_date DATE) RETURN NUMBER IS
                                 
  v_success NUMBER(1);
  v_date    DATE;
  
  CURSOR c_get_date IS
    SELECT arrival_date FROM tsrstvst
    WHERE tsmstatn_is_number = p_statn_is_number
    AND   id_number          = p_visit_id;
  
BEGIN

  OPEN c_get_date;
  FETCH c_get_date INTO v_date;
    IF c_get_date%NOTFOUND THEN
      v_success := 1;
      RETURN v_success;
    END IF;
  CLOSE c_get_date;
  
  IF TO_CHAR(v_date,'MM/DD/RRRR') = TO_CHAR(p_visit_start_date,'MM/DD/RRRR') THEN
    v_success := 1;
  ELSE
    v_success := 2;
  END IF;
  
  RETURN v_success;
  
EXCEPTION
  WHEN OTHERS THEN
     Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in check visit start date.',NULL,NULL,NULL);
     v_success := 2;
     RETURN v_success;
END check_visit_start_date;

/******************************************************************************************************/
PROCEDURE get_next_stavst_id (p_stavst_id OUT VARCHAR2) IS

BEGIN
  /* kms (6/2/01) - this function will get the next station visit number.  This is called
     only when the instructions are to create the station on import.  This will take the
     last station visit id number used by the import and increment it by 1. This station
     visit id will always be 'UNA' and the resulting number for all visits created by SIM.
     The actual visit id will be determined upon export. */

  /* kms - get visit id by incrementing the last one used by 1 */
  v_last_visit_id := v_last_visit_id + 1;

  p_stavst_id := 'UNA'||TO_CHAR(v_last_visit_id);

EXCEPTION
   WHEN OTHERS THEN
     Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in get next visit id.',NULL,NULL,NULL);

END get_next_stavst_id;

/******************************************************************************************************/
PROCEDURE check_existing_visit (p_instr_choice IN VARCHAR2, p_act_start_date IN DATE,
                                p_act_start_time IN DATE, p_trip_seq IN NUMBER,
                                p_station_key IN t_storet_pk_type, p_visit_start_date OUT DATE,
                                p_visit_start_time OUT DATE, p_visit_end_date OUT DATE,
                                p_visit_end_time OUT DATE, p_visit_key OUT t_storet_pk_type,
                                p_visit_seq IN OUT NUMBER, p_visit_id_tab IN OUT t_stvst_id_type,
                                p_visit_id IN OUT VARCHAR2) IS

  v_success                       NUMBER(1);

  
  CURSOR c_find_station_visit_dt(p_stat_is_nbr IN NUMBER, p_stat_org_id IN VARCHAR2,
                                 p_trip_seq IN NUMBER, p_date IN DATE, p_time IN DATE) IS
    SELECT SSV_SEQ, SSV_TSRSTVST_IS_NUMBER, SSV_TSRSTVST_ORG_ID FROM SIM_STATION_VISITS
      WHERE TO_CHAR(SSV_START_TIME,'HH:MI:SS') = 
      TO_CHAR(NVL(p_time,TO_DATE('00:01','HH24:MI')),'HH:MI:SS')
      AND TRUNC(SSV_START_DATE,'DD') = TRUNC(p_date,'DD')
      AND SSV_TSMSTATN_IS_NUMBER = p_stat_is_nbr
      AND SSV_TSMSTATN_ORG_ID = RPAD(p_stat_org_id,8) 
      AND SSV_STP_SEQ = p_trip_seq;


  CURSOR c_find_station_visit(p_stat_is_nbr IN NUMBER, p_stat_org_id IN VARCHAR2,
                              p_trip_seq IN NUMBER, p_date IN DATE) IS
    SELECT SSV_SEQ, SSV_TSRSTVST_IS_NUMBER, SSV_TSRSTVST_ORG_ID FROM SIM_STATION_VISITS
      WHERE TRUNC(SSV_START_DATE,'DD') = TRUNC(p_date,'DD') AND SSV_TSMSTATN_IS_NUMBER =  p_stat_is_nbr
      AND SSV_TSMSTATN_ORG_ID = RPAD(p_stat_org_id,8) AND SSV_STP_SEQ = p_trip_seq;

BEGIN
  /* kms (5/31/01) - function called when the instructions are to create the station visit.
     Checks the sim_station_visits table to see if a station visit already exists for the
     trip, station, date, and time of the activity's station visit.  Date and time are based
     on the instructions.  If not, will look for station visit in STORET and create a new
     record in sim_station_visits table whether or not the visit already exists in STORET.
     IF it does exist, the 'created by sim' flag will be set accordingly and the foreign key
     columns to TSRSTVST filled in. */

  IF p_instr_choice = 'RES-STVST-01' THEN
    /* kms - one station visit per day.  Check sim_station_visits to see if visit
       already exists for this trip, station, and date. */
    OPEN c_find_station_visit (p_station_key.is_number, p_station_key.org_id,
                               p_trip_seq, p_act_start_date);
    FETCH c_find_station_visit INTO p_visit_seq, p_visit_key.is_number, p_visit_key.org_id;
      IF c_find_station_visit%NOTFOUND THEN
        p_visit_seq := NULL;
        p_visit_key.is_number := NULL;
        p_visit_key.org_id := NULL;
      END IF;
    CLOSE c_find_station_visit;
/* sef 9/25/01 - don't want to create unsolicited info in STORET so changing the lines
                 for end date, start time, and end time */
    p_visit_start_date := p_act_start_date;
    p_visit_end_date := NULL;
    p_visit_start_time := NULL;
    p_visit_end_time := NULL;

--    p_visit_end_date := p_act_start_date;
--    p_visit_start_time := TO_DATE('00:01','HH24:MI');
--    p_visit_end_time := TO_DATE('23:59','HH24:MI');

    ELSIF p_instr_choice = 'RES-STVST-03' THEN
    /* kms - one station visit per date/time.  Check sim_station_visits to see if
       visit already exists for this trip, station, date, and time. */
    OPEN c_find_station_visit_dt (p_station_key.is_number, p_station_key.org_id,
                                  p_trip_seq, p_act_start_date, p_act_start_time);
    FETCH c_find_station_visit_dt INTO p_visit_seq, p_visit_key.is_number, p_visit_key.org_id;
      IF c_find_station_visit_dt%NOTFOUND THEN
        p_visit_seq := NULL;
        p_visit_key.is_number := NULL;
        p_visit_key.org_id := NULL;
      END IF;
    CLOSE c_find_station_visit_dt;
/* sef 9/25/01 - don't want to create unsolicited info in STORET so changing the lines
                 for end date, start time, and end time */
    p_visit_start_date := p_act_start_date;
    p_visit_end_date := NULL;
--    p_visit_start_time := NULL;
    p_visit_end_time := NULL;

--    p_visit_end_date := p_act_start_date;
    p_visit_start_time := NVL(p_act_start_time,TO_DATE('00:01','HH24:MI'));
--    p_visit_end_time := TO_DATE('23:59','HH24:MI');

  END IF;

  
  IF p_visit_seq IS NULL THEN
    /* kms - station visit does not already exist in SIM or in STORET, get visit id.
       If the visit id was not obtained from TSRSTVST, get it by counting the number
       of station visits that already exist on the trip for the station and adding 1. */

    IF p_visit_id IS NULL THEN
      get_next_stavst_id(p_visit_id);
    END IF;

  END IF;

EXCEPTION
   WHEN OTHERS THEN
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in check existing visit.',NULL,NULL,NULL);

END check_existing_visit;

/******************************************************************************************************/
FUNCTION create_field_activity (p_fa_rec IN SIM_FIELD_ACTIVITIES%ROWTYPE) RETURN NUMBER IS

BEGIN

  /* create field activity using values in record */
  INSERT INTO SIM_FIELD_ACTIVITIES (
    FA_SEQ,
    FA_TYPE_SEQ,
    FA_CATEGORY_SEQ,
    FA_ACTIVITY_ID,
    FA_START_DATE,
    FA_START_TIME,
    FA_STOP_DATE,
    FA_STOP_TIME,
    FA_DEPTH,
    FA_DEPTH_REF_POINT_SEQ,
    FA_ACTIVITY_COMMENTS,
    FA_TSMSTATN_IS_NUMBER,
    FA_TSMSTATN_ORG_ID,
    FA_TSRTRIP_IS_NUMBER,
    FA_TSRTRIP_ORG_ID,
    FA_TSRSTVST_IS_NUMBER,
    FA_TSRSTVST_ORG_ID,
    FA_TSRFLDPR_IS_NUMBER,
    FA_TSRFLDPR_ORG_ID,
    FA_TSRFLDGR_IS_NUMBER,
    FA_TSRFLDGR_ORG_ID,
    FA_IMPORT_SEQ,
    FA_EXPORT_SEQ,
    FA_STATUS,
    FA_TSMORGAN_IS_NUMBER,
    FA_TSMORGAN_ORG_ID,
    FA_APPROVED_BY,
    FA_APPROVED_ON,
    FA_STVST_COMMENTS,
    FA_DU_SEQ,
    FA_MED_SEQ,
    FA_TSRLAB_IS_NUMBER,
    FA_TSRLAB_ORG_ID,
    FA_IMPORT_LINE_NO,
    FA_TSRFDACT_IS_NUMBER,
    FA_TSRFDACT_ORG_ID,
    FA_REPLICATE_NBR,
    FA_FDSET_ID,
    FA_FDSET_NAME,
    FA_UPDEPTH,
    FA_LOWDEPTH,
    FA_THERMO_SEQ,
    FA_HALO_SEQ,
    FA_PYCNO_SEQ,
    FA_REL_DEPTH_SEQ,
    FA_DEPMEASFROM_SEQ,
    FA_DEPTHZONE_SEQ,
    FA_DEPTHRU_SEQ,
    FA_TIME_ZONE_SEQ,
    FA_TSRGRCFG_IS_NUMBER,
    FA_TSRGRCFG_ORG_ID,
    FA_TSRSDP_IS_NUMBER,
    FA_TSRSDP_ORG_ID,
    FA_STP_SEQ,
    FA_SSV_SEQ
    )
  VALUES(
    p_fa_rec.FA_SEQ,
    p_fa_rec.FA_TYPE_SEQ,
    p_fa_rec.FA_CATEGORY_SEQ,
    p_fa_rec.FA_ACTIVITY_ID,
    p_fa_rec.FA_START_DATE,
    p_fa_rec.FA_START_TIME,
    p_fa_rec.FA_STOP_DATE,
    p_fa_rec.FA_STOP_TIME,
    p_fa_rec.FA_DEPTH,
    p_fa_rec.FA_DEPTH_REF_POINT_SEQ,
    p_fa_rec.FA_ACTIVITY_COMMENTS,
    p_fa_rec.FA_TSMSTATN_IS_NUMBER,
    p_fa_rec.FA_TSMSTATN_ORG_ID,
    p_fa_rec.FA_TSRTRIP_IS_NUMBER,
    p_fa_rec.FA_TSRTRIP_ORG_ID,
    p_fa_rec.FA_TSRSTVST_IS_NUMBER,
    p_fa_rec.FA_TSRSTVST_ORG_ID,
    p_fa_rec.FA_TSRFLDPR_IS_NUMBER,
    p_fa_rec.FA_TSRFLDPR_ORG_ID,
    p_fa_rec.FA_TSRFLDGR_IS_NUMBER,
    p_fa_rec.FA_TSRFLDGR_ORG_ID,
    p_fa_rec.FA_IMPORT_SEQ,
    p_fa_rec.FA_EXPORT_SEQ,
    p_fa_rec.FA_STATUS,
    p_fa_rec.FA_TSMORGAN_IS_NUMBER,
    p_fa_rec.FA_TSMORGAN_ORG_ID,
    p_fa_rec.FA_APPROVED_BY,
    p_fa_rec.FA_APPROVED_ON,
    p_fa_rec.FA_STVST_COMMENTS,
    p_fa_rec.FA_DU_SEQ,
    p_fa_rec.FA_MED_SEQ,
    NULL, --p_fa_rec.FA_TSRLAB_IS_NUMBER,
    NULL, --p_fa_rec.FA_TSRLAB_ORG_ID,
    p_fa_rec.FA_IMPORT_LINE_NO,
    p_fa_rec.FA_TSRFDACT_IS_NUMBER,
    p_fa_rec.FA_TSRFDACT_ORG_ID,
    p_fa_rec.FA_REPLICATE_NBR,
    p_fa_rec.FA_FDSET_ID,
    p_fa_rec.FA_FDSET_NAME,
    p_fa_rec.FA_UPDEPTH,
    p_fa_rec.FA_LOWDEPTH,
    p_fa_rec.FA_THERMO_SEQ,
    p_fa_rec.FA_HALO_SEQ,
    p_fa_rec.FA_PYCNO_SEQ,
    p_fa_rec.FA_REL_DEPTH_SEQ,
    p_fa_rec.FA_DEPMEASFROM_SEQ,
    p_fa_rec.FA_DEPTHZONE_SEQ,
    p_fa_rec.FA_DEPTHRU_SEQ,
    p_fa_rec.FA_TIME_ZONE_SEQ,
    p_fa_rec.FA_TSRGRCFG_IS_NUMBER,
    p_fa_rec.FA_TSRGRCFG_ORG_ID,
    p_fa_rec.FA_TSRSDP_IS_NUMBER,
    p_fa_rec.FA_TSRSDP_ORG_ID,
    p_fa_rec.FA_STP_SEQ,
    p_fa_rec.FA_SSV_SEQ
    );

  COMMIT;

  RETURN 1;

EXCEPTION

  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - inserting field activity.',NULL,p_fa_rec.fa_seq,NULL);
    RETURN 0;

END create_field_activity;


/******************************************************************************************************/
FUNCTION add_project_to_activity(p_fa_seq IN NUMBER, p_project_is_nbr IN NUMBER,
                                   p_project_org IN VARCHAR2) RETURN NUMBER IS

  v_act_project_seq                  NUMBER(10);

  CURSOR c_get_act_project_seq IS
    SELECT SIM_ACTIVITY_PROJECTS_SEQ.NEXTVAL FROM DUAL;

BEGIN
  /* kms - function that associates projects with activities in SIM */

  /* get next sequence */
  OPEN c_get_act_project_seq;
  FETCH c_get_act_project_seq INTO v_act_project_seq;
    IF c_get_act_project_seq%NOTFOUND THEN
      v_act_project_seq := NULL;
    END IF;
  CLOSE c_get_act_project_seq;

  INSERT INTO SIM_ACTIVITY_PROJECTS (
    SAPJ_SEQ,
    SAPJ_FA_SEQ,
    SAPJ_TSMPROJ_IS_NUMBER,
    SAPJ_TSMPROJ_ORG_ID
    )
  VALUES (
    v_act_project_seq,               /* SAPJ_SEQ */
    p_fa_seq,                        /* SAPJ_FA_SEQ */
    p_project_is_nbr,                /* SAPJ_TSMPROJ_IS_NUMBER */
    p_project_org                    /* SAPJ_TSMPROJ_ORG_ID */
    );

  COMMIT;

  RETURN 1;

EXCEPTION

  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in add project to activity.',NULL,p_fa_seq,NULL);
    RETURN 0;

END add_project_to_activity;

/******************************************************************************************************/
FUNCTION add_personnel_to_activity(p_fa_seq IN NUMBER, p_persn_is_nbr IN NUMBER,
                                   p_persn_org IN VARCHAR2) RETURN NUMBER IS

  v_act_persn_seq                  NUMBER(10);

  CURSOR c_get_act_persn_seq IS
    SELECT SIM_ACTIVITY_PERSONNEL_SEQ.NEXTVAL FROM DUAL;

BEGIN
  /* kms - function that associates personnel with activities in SIM. */

  /* get next sequence */
  OPEN c_get_act_persn_seq;
  FETCH c_get_act_persn_seq INTO v_act_persn_seq;
  CLOSE c_get_act_persn_seq;

  INSERT INTO SIM_ACTIVITY_PERSONNEL (
    SAP_SEQ,
    SAP_FA_SEQ,
    SAP_TSMPERSN_IS_NUMBER,
    SAP_TSMPERSN_ORG_ID
    )
  VALUES (
    v_act_persn_seq,                 /* SAP_SEQ */
    p_fa_seq,                        /* SAP_FA_SEQ */
    p_persn_is_nbr,                  /* SAP_TSMPERSN_IS_NUMBER */
    p_persn_org                      /* SAP_TSMPERSN_ORG_ID */
    );

  COMMIT;

  RETURN 1;

EXCEPTION

  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in add personnel to activity.',NULL,p_fa_seq,NULL);
    RETURN 0;

END add_personnel_to_activity;

/******************************************************************************************************/
FUNCTION check_group_type_med(p_group_pk IN t_storet_pk_type, p_type_seq IN NUMBER, p_medium_seq IN NUMBER,
                              p_fa_seq IN NUMBER) RETURN NUMBER IS

  v_group_type              VARCHAR2(60);
  v_group_medium            VARCHAR2(60);
  v_act_type                VARCHAR2(13);
  v_act_medium              VARCHAR2(20);

  v_return_value      NUMBER(1);

  CURSOR c_get_group_type_med(p_group_pk IN t_storet_pk_type) IS
    SELECT FLD_ACT_TYPE_NM, MEDIUM_TYPE_NAME FROM TSRCHGRP
      WHERE TSRCHGRP_IS_NUMBER = p_group_pk.is_number AND TSRCHGRP_ORG_ID = p_group_pk.org_id;

  CURSOR c_get_alval_item(p_sia_seq IN NUMBER) IS
    SELECT SIA_ITEM FROM SIM_IMP_ALLOW_VALUES WHERE SIA_SEQ = p_sia_seq;

BEGIN
  /* kms (5/25/01) - function that checks the characterstic group activity type and medium
     against those on the activity being imported.  If they are the same, the group
     is valid and the function returns a 1, otherwise it will return a 2 and an error
     message will be written in the main program. */

  /* kms - get the activity type and medium from the characteristic group */
  OPEN c_get_group_type_med(p_group_pk);
  FETCH c_get_group_type_med INTO v_group_type, v_group_medium;
    IF c_get_group_type_med%FOUND THEN

      /* kms - get medium and activity type values for the activity to compare to those
         for the group. */
      OPEN c_get_alval_item(p_type_seq);
      FETCH c_get_alval_item INTO v_act_type;
      CLOSE c_get_alval_item;

      OPEN c_get_alval_item(p_medium_seq);
      FETCH c_get_alval_item INTO v_act_medium;
      CLOSE c_get_alval_item;

      /* kms - compare group and activity medium and type */
      /* bka - added check for Trip QC that allows 'Sample' char groups for activities of type Trip QC */
      IF RTRIM(v_act_type) = 'Trip QC' THEN
        IF RTRIM(v_group_type) != 'Sample' OR RPAD(v_group_medium,20) != RPAD(v_act_medium,20) THEN
          v_return_value := 2;
          Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Characteristic group does not have the same activity type and medium as the activity for this result.',NULL,p_fa_seq,NULL);
        ELSE
          v_return_value := 1;
        END IF;
      ELSIF RPAD(v_group_type,13) != RPAD(v_act_type,13) OR RPAD(v_group_medium,20) != RPAD(v_act_medium,20) THEN
        v_return_value := 2;
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Characteristic group does not have the same activity type and medium as the activity for this result.',NULL,p_fa_seq,NULL);
      ELSE
        v_return_value := 1;
      END IF;

    ELSE
      v_return_value := 2;
      Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Error getting characteristic group activity type and medium - unable to compare these to values for activity.',NULL,p_fa_seq,NULL);
    END IF;
  CLOSE c_get_group_type_med;

  RETURN v_return_value;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in check group type med.',NULL,p_fa_seq,NULL);
    RETURN 0;

END check_group_type_med;

/******************************************************************************************************/
FUNCTION get_analysis_code_seq(p_activity_seq IN NUMBER,p_analysis_code IN VARCHAR2,p_status IN OUT VARCHAR2,
                               p_tsrlab_is_nbr IN NUMBER, p_tsrlab_org_id IN VARCHAR2,
                               p_line_nbr IN NUMBER, p_fa_seq IN NUMBER) RETURN NUMBER IS

BEGIN
   NULL;

/* DTW 09/17/01 Not supporting Analysis Codes in this version

  v_analysis_code        VARCHAR2(15);
  v_analysis_seq         NUMBER(10);

  CURSOR c_analysis_id (p_analysis_id IN VARCHAR2) IS
    SELECT AC_SEQ FROM SIM_ANALYSIS_CODES WHERE AC_ANALYSIS_ID = p_analysis_id;

  CURSOR c_analysis_seq IS
    SELECT SIM_ANALYSIS_CODES_SEQ.NEXTVAL FROM DUAL;


BEGIN
  -- kms - gets analysis code sequence using the processed analysis id

  -- figure out whether to do an upper on the analysis code from the instructions. For now,
  -- do it for UT data
     null;
  v_analysis_code := UPPER(p_analysis_code);

  OPEN c_analysis_id(v_analysis_code);
  FETCH c_analysis_id INTO v_analysis_seq;
    BEGIN
      IF c_analysis_id%NOTFOUND THEN
        -- if analysis code is not in the table, update status to 'N' and create record
        p_status := 'N';

        -- get analysis code sequence
        OPEN c_analysis_seq;
        FETCH c_analysis_seq INTO v_analysis_seq;
        CLOSE c_analysis_seq;

        BEGIN
          INSERT INTO SIM_ANALYSIS_CODES (
            AC_SEQ,
            AC_ANALYSIS_ID,
            AC_TSRLAB_IS_NUMBER,
            AC_TSRLAB_ORG_ID,
            AC_DESC,
            AC_STATUS
            )
          VALUES (
            v_analysis_seq,                                 -- AC_SEQ
            p_analysis_code,                                -- AC_ANALYSIS_ID
            p_tsrlab_is_nbr,                                -- AC_TSRLAB_IS_NUMBER
            p_tsrlab_org_id,                                -- AC_TSRLAB_ORG_ID
            'created by import '||TO_CHAR(v_import_log_seq),-- AC_DESC
            'B'                                             -- AC_STATUS
            );

          COMMIT;
        EXCEPTION
          WHEN OTHERS THEN
            Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - Inserting new analysis ID seq. Line: '||p_line_nbr,NULL,p_fa_seq,NULL);
        END;

      END IF;
    EXCEPTION
      WHEN OTHERS THEN
        Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - getting new analysis ID seq. Line: '||p_line_nbr,NULL,p_fa_seq,NULL);
    END;
  CLOSE c_analysis_id;

  p_status := p_status;

  RETURN v_analysis_seq;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in get analysis code sequence.',NULL,p_fa_seq,NULL);
    RETURN NULL;
*** DTW ***/
END get_analysis_code_seq;

/******************************************************************************************************/
FUNCTION check_chardet_comp(p_char_is_nbr IN NUMBER, p_char_org_id IN VARCHAR2,
                            p_anlpr_is_nbr IN NUMBER, p_uom_is_nbr IN NUMBER,
                            p_uom_org_id IN VARCHAR2, p_fraction IN VARCHAR2,
                            p_value_type IN NUMBER, p_units_ok IN BOOLEAN,
                            p_fa_seq IN NUMBER) RETURN NUMBER IS

  v_proc_req                VARCHAR2(1);
  v_samp_req                VARCHAR2(1);
  v_uom_type                VARCHAR2(5);
  v_uom_uom_type            VARCHAR2(5);
  v_char                    VARCHAR2(60);
  v_return_value            NUMBER(1) := 1;

  e_bad_char_pk             EXCEPTION;

  CURSOR c_get_char_reqs(p_char_is_number IN NUMBER, p_char_org_id IN VARCHAR2) IS
    SELECT PROC_REQ_IND_CD, SAMP_FRAC_REQ_CD, UOM_TYPE, DISPLAY_NAME FROM TSRCHAR
      WHERE TSRCHAR_IS_NUMBER = p_char_is_number AND TSRCHAR_ORG_ID = RPAD(p_char_org_id, 8);

  CURSOR c_get_uom_type(p_uom_is_nbr IN NUMBER, p_uom_org_id IN VARCHAR2) IS
    SELECT UOM_TYPE FROM TSRUOM WHERE TSRUOM_IS_NUMBER = p_uom_is_nbr
      AND TSRUOM_ORG_ID = p_uom_org_id;


BEGIN
  /* kms - function that checks characteristic requirements against what has been provided
     for the result being imported */

  /* get the requirements for this result's characteristic in order to make sure that they have
     been met */
  OPEN c_get_char_reqs(p_char_is_nbr, p_char_org_id);
  FETCH c_get_char_reqs INTO v_proc_req, v_samp_req, v_uom_type, v_char;
    IF c_get_char_reqs%NOTFOUND THEN
      /* if nothing found then char pk is bogus */
      CLOSE c_get_char_reqs;
      RAISE e_bad_char_pk;
    END IF;
  CLOSE c_get_char_reqs;

  /* make sure value type is not null */
  IF p_value_type IS NULL AND v_char_result_type NOT IN ('TEXT') THEN  -- dcl If the char type is 'Text' then it's  OK not to have a value type
    v_return_value := 3;
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning: Value type is required for all results.',NULL,p_fa_seq,NULL);
  END IF;

  /* check analysis procedure */
  IF v_proc_req = 'Y' AND p_anlpr_is_nbr IS NULL THEN
    v_return_value := 2;
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Field/Lab Procedure has not been provided and is required for results with the characteristic '||v_char,NULL,p_fa_seq,NULL);
  END IF;

  /* check sample fraction */
  IF v_samp_req = 'Y' AND p_fraction IS NULL THEN
    v_return_value := 2;
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Sample fraction has not been provided and is required for results with the characteristic '||v_char,NULL,p_fa_seq,NULL);
  END IF;

  /* check units and their type */
  IF v_uom_type = 'NONE ' AND p_uom_is_nbr IS NOT NULL THEN
    /* kms - units are not required, but have been provided, issue a warning. */
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning: Characteristic, '||v_char||', does not require a unit of measure but one has been provided.',NULL,p_fa_seq,NULL);

  ELSIF NOT p_units_ok AND p_uom_is_nbr IS NOT NULL THEN
    /* kms (5/29/01) - units should not be provided since this result is a
       non-detect (or similar).  Warn user that units will be removed from the
       result and set flag to do so. */
    v_return_value := 3;
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning: Result value has not been reported, unit of measure removed from result.',NULL,p_fa_seq,NULL);

  ELSIF v_uom_type != 'NONE ' AND p_uom_is_nbr IS NULL AND p_units_ok THEN
    /* unit of measure is required but has not been provided */
    v_return_value := 2;
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Unit of Measure has not been provided and is required for results with the characteristic: '||v_char,NULL,p_fa_seq,NULL);

  ELSIF v_uom_type != 'NONE ' AND p_units_ok THEN
    /* if unit of measure is required and has been provided, make sure acode and char
       uom types match */
    OPEN c_get_uom_type(p_uom_is_nbr, p_uom_org_id);
    FETCH c_get_uom_type INTO v_uom_uom_type;
      IF c_get_uom_type%NOTFOUND THEN
        /* shouldn't happen since we have an ac_seq and an is number */
        RAISE_APPLICATION_ERROR(-20100,' - c_get_uom_type did not find a value.');
      ELSE
        IF v_uom_type != v_uom_uom_type THEN
          /* unit of measure has been provided but does not match the char uom */
          v_return_value := 2;
          Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Result with the characteristic, '||v_char||' has the wrong type of unit for this characteristic.',NULL,p_fa_seq,NULL);
        END IF;
      END IF;
    CLOSE c_get_uom_type;

  END IF;

  RETURN v_return_value;

EXCEPTION
  WHEN OTHERS THEN
  Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in check_chardet_comp.',NULL,p_fa_seq,NULL);
  RETURN 0;

END check_chardet_comp;

/******************************************************************************************************/
FUNCTION check_acode_comp(p_ac_seq IN NUMBER, p_fa_seq IN NUMBER) RETURN NUMBER IS

BEGIN
  NULL;
/* DTW 09/17/01 Analysis Codes not supported in this version of SIM
  v_comp_cd                        VARCHAR2(1);
  v_analysis_code                  VARCHAR2(15);
  v_char_is_nbr                    NUMBER(8);
  v_char_org_id                    VARCHAR2(8);
  v_uom_is_nbr                     NUMBER(8);
  v_uom_org_id                     VARCHAR2(8);
  v_anlpr_is_nbr                   NUMBER(8);
  v_fraction                       VARCHAR2(15);
  v_value_type                     VARCHAR2(10);
  v_return_value                   NUMBER(1) := 1;

  CURSOR c_get_comp_cd(p_ac_seq IN NUMBER) IS
    SELECT AC_ANALYSIS_ID, COMP_IND_CD FROM TSRCHDEF, SIM_ANALYSIS_CODES
      WHERE ac_seq = p_ac_seq
      AND ac_tsrchdef_is_number = tsrchdef_is_number
      AND ac_tsrchdef_org_id = tsrchdef_org_id;

  CURSOR c_check_acode(p_ac_seq IN NUMBER) IS
    SELECT AC_ANALYSIS_ID, AC_TSRCHAR_IS_NUMBER, AC_TSRCHAR_ORG_ID, AC_TSRUOM_IS_NUMBER, AC_TSRUOM_ORG_ID,
      AC_TSRANLPR_IS_NUMBER, AC_SAMPLE_FRAC, AC_VALUE_TYPE
      FROM SIM_ANALYSIS_CODES, TSRCHAR
      WHERE AC_SEQ = p_ac_seq
      AND AC_TSRCHAR_IS_NUMBER = TSRCHAR_IS_NUMBER
      AND AC_TSRCHAR_ORG_ID = AC_TSRCHAR_ORG_ID;

  CURSOR c_get_uom_type(p_ac_seq IN NUMBER) IS
    SELECT UOM_TYPE FROM TSRUOM, SIM_ANALYSIS_CODES WHERE AC_SEQ = p_ac_seq
      AND AC_TSRUOM_IS_NUMBER = TSRUOM_IS_NUMBER AND AC_TSRUOM_ORG_ID = TSRUOM_ORG_ID;

BEGIN
  -- kms - function that checks the analysis code to make sure it's complete.

  -- try to get comp ind code from STORET table, if nothing found,
  -- assume char def comes from sim_analysis_codes
  OPEN c_get_comp_cd(p_ac_seq);
  FETCH c_get_comp_cd INTO v_analysis_code, v_comp_cd;
    IF c_get_comp_cd%NOTFOUND THEN

      -- cursor did not return value so assume char info comes directly from sim_analysis_codes
      OPEN c_check_acode(p_ac_seq);
      FETCH c_check_acode INTO v_analysis_code,v_char_is_nbr, v_char_org_id, v_uom_is_nbr, v_uom_org_id, v_anlpr_is_nbr, v_fraction, v_value_type;
        IF c_check_acode%NOTFOUND THEN
          -- kms (5/29/01) - analysis code has not been mapped to either a STORET characteristic
          -- row or a characteristic - write an error message and do not allow export of the
          -- activity.
          IF v_char_is_nbr IS NULL THEN
            v_return_value := 2;
            Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Analysis Code, '||v_analysis_code||', has not been associated with either a characteristic or a STORET group and row.',NULL,p_fa_seq,NULL);
          END IF;
        ELSE
          -- kms (5/29/01)- commented out since this function was being run twice when an
          -- analysis code was used.  If analysis code has been associated with a characteristic
          -- (it has if the cursor returns a value) then the defaults will be checked later
          -- in the program.

          -- run function to check required fields
          -- kms v_return_value := check_chardet_comp(v_char_is_nbr, v_char_org_id, v_anlpr_is_nbr,
          --                                   v_uom_is_nbr, v_uom_org_id, v_fraction,
          --                                   v_value_type, p_fa_seq);

          NULL;
         END IF;
      CLOSE c_check_acode;

    ELSE
      -- if row not complete, write error and set flag to prevent export
      IF v_comp_cd = 'N' THEN
        v_return_value := 2;
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Analysis Code, '||v_analysis_code||', is assigned to an incomplete Characteristic Row.',NULL,p_fa_seq,NULL);
      END IF;
    END IF;
  CLOSE c_get_comp_cd;

  RETURN v_return_value;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in check acode comp.',NULL,p_fa_seq,NULL);
    RETURN 0;
 *** DTW ***/
END check_acode_comp;

/******************************************************************************************************/

FUNCTION get_char_result_type(p_char_is_nbr IN NUMBER) RETURN VARCHAR2 IS

v_char_result_type VARCHAR2(4);

  CURSOR c_get_char_type(p_char_is_nbr IN NUMBER) IS
    SELECT D_SCR_TYPE_CD FROM TSRCHAR
      WHERE TSRCHAR_IS_NUMBER = p_char_is_nbr
      AND TSRCHAR_ORG_ID = 'ZZZZZZZZ';

BEGIN
  /* kms - function that gets characteristic type in order to see if it has permitted values
     or if other checks need to be performed on the result. */

  OPEN c_get_char_type(p_char_is_nbr);
  FETCH c_get_char_type INTO v_char_result_type;
    IF c_get_char_type%NOTFOUND THEN
      v_char_result_type := NULL;
    END IF;
  CLOSE c_get_char_type;

RETURN v_char_result_type;

END get_char_result_type;

/******************************************************************************************************/
FUNCTION get_col_seq(p_col_name IN VARCHAR2, p_imp_cfg_seq IN NUMBER, p_fa_seq IN NUMBER) RETURN NUMBER IS

  v_col_seq NUMBER(10);

  CURSOR c_get_col_seq(cp_col_name IN VARCHAR2, cp_imp_cfg_seq IN NUMBER) IS
    SELECT SIC_SEQ FROM SIM_IMP_COLS, SIM_IMP_CFG_DETS, SIM_IMP_CFG
    WHERE SIC_SEQ  = SICDT_SIC_SEQ
    AND   SICC_SEQ = SICDT_SICC_SEQ
    AND   SICC_SEQ = cp_imp_cfg_seq
    AND   SIC_NAME = cp_col_name;

BEGIN

  OPEN c_get_col_seq(p_col_name, p_imp_cfg_seq);
  FETCH c_get_col_seq INTO v_col_seq;
  CLOSE c_get_col_seq;

  RETURN v_col_seq;

  EXCEPTION WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - In get col seq.',NULL,p_fa_seq,NULL);
    RETURN NULL;

END get_col_seq;

/******************************************************************************************************/
FUNCTION get_char_defaults(p_analysis_seq IN NUMBER, p_row_pk IN t_storet_pk_type, p_imp_cfg_seq IN NUMBER,
                           p_col_seq IN NUMBER, p_fa_seq IN NUMBER) RETURN NUMBER IS

  v_dummy                  NUMBER(8) := NULL;
  v_uom_name               VARCHAR2(10);
  v_dur_basis              VARCHAR2(10);
  v_stat_type              VARCHAR2(18);
  v_value_type             VARCHAR2(10);
  v_temp_basis             VARCHAR2(8);
  v_col_seq                NUMBER(10);

  e_bad_ac_seq             EXCEPTION;
  e_bad_def_map            EXCEPTION;
  e_bad_def_pk             EXCEPTION;

  CURSOR c_check_acode_type(p_analysis_seq IN NUMBER) IS
    SELECT AC_TSRCHDEF_IS_NUMBER FROM SIM_ANALYSIS_CODES WHERE AC_SEQ = p_analysis_seq;

  CURSOR c_get_acode_defs(p_analysis_seq IN NUMBER) IS
    SELECT AC_SAMPLE_FRAC fraction, AC_TSRUOM_IS_NUMBER uom_is_nbr, AC_TSRUOM_ORG_ID uom_org_id,
      AC_DURATION_BASIS dur_basis, AC_STATISTIC_TYPE stat_type, AC_VALUE_TYPE value_type,
      AC_TEMP_BASIS temp_basis, AC_TSRANLPR_IS_NUMBER anlpr_is_nbr, AC_TSRANLPR_ORG_ID anlpr_org_id,
      AC_TSRLSPP_IS_NUMBER lspp_is_nbr, AC_TSRLSPP_ORG_ID lspp_org_id, AC_TSRCHAR_IS_NUMBER char_is_nbr,
      AC_TSRCHAR_ORG_ID char_org_id, AC_TSRLAB_IS_NUMBER lab_is_nbr, AC_TSRLAB_ORG_ID lab_org_id
      FROM SIM_ANALYSIS_CODES WHERE AC_SEQ = p_analysis_seq;

  CURSOR c_get_storet_row_defs(p_row_pk IN t_storet_pk_type) IS
    SELECT DEF.SMPL_FRAC_TYPE_NM fraction, DEF.UOM_NAME uom_name, DEF.DUR_BASIS_TYPE_NM dur_basis,
      DEF.STATISTIC_TYPE_NM stat_type, DEF.VALUE_TYPE_NAME value_type, DEF.TEMP_BASIS_LVL_NM temp_basis,
      DEF.TSRANLPR_IS_NUMBER anlpr_is_nbr, DEF.TSRANLPR_ORG_ID anlpr_org_id, DEF.TSRLSPP_IS_NUMBER lspp_is_number,
      DEF.TSRLSPP_ORG_ID lspp_org_id, CCGA.TSRCHAR_IS_NUMBER char_is_nbr, CCGA.TSRCHAR_ORG_ID char_org_id
      FROM TSRCHDEF DEF, TSRCCGA CCGA
      WHERE DEF.TSRCCGA_IS_NUMBER = CCGA.TSRCCGA_IS_NUMBER
      AND DEF.TSRCCGA_ORG_ID = CCGA.TSRCCGA_ORG_ID
      AND DEF.TSRCHDEF_IS_NUMBER = p_row_pk.is_number
      AND DEF.TSRCHDEF_ORG_ID = RPAD(p_row_pk.org_id,8);

  CURSOR c_get_storet_defs(p_analysis_seq IN NUMBER) IS
    SELECT DEF.SMPL_FRAC_TYPE_NM fraction, DEF.UOM_NAME uom_name, DEF.DUR_BASIS_TYPE_NM dur_basis,
      DEF.STATISTIC_TYPE_NM stat_type, DEF.VALUE_TYPE_NAME value_type, DEF.TEMP_BASIS_LVL_NM temp_basis,
      DEF.TSRANLPR_IS_NUMBER anlpr_is_nbr, DEF.TSRANLPR_ORG_ID anlpr_org_id, DEF.TSRLSPP_IS_NUMBER lspp_is_number,
      DEF.TSRLSPP_ORG_ID lspp_org_id, CCGA.TSRCHAR_IS_NUMBER char_is_nbr, CCGA.TSRCHAR_ORG_ID char_org_id,
      AC_TSRLAB_IS_NUMBER lab_is_nbr, AC_TSRLAB_ORG_ID lab_org_id
      FROM SIM_ANALYSIS_CODES, TSRCHDEF DEF, TSRCCGA CCGA
      WHERE AC_TSRCHDEF_IS_NUMBER = DEF.TSRCHDEF_IS_NUMBER
      AND AC_TSRCHDEF_ORG_ID = DEF.TSRCHDEF_ORG_ID
      AND DEF.TSRCCGA_IS_NUMBER = CCGA.TSRCCGA_IS_NUMBER
      AND DEF.TSRCCGA_ORG_ID = CCGA.TSRCCGA_ORG_ID
      AND AC_SEQ = p_analysis_seq;

  CURSOR c_get_col_seq(p_col_name IN VARCHAR2) IS
    SELECT SIC_SEQ FROM SIM_IMP_COLS WHERE SIC_NAME = p_col_name;

BEGIN
  /* kms - function that gets characteristic defaults from STORET to be stored with results in SIM */

  /* see if analysis code has been mapped to STORET group and row */
  IF p_analysis_seq IS NOT NULL THEN
    OPEN c_check_acode_type(p_analysis_seq);
    FETCH c_check_acode_type INTO v_dummy;
      IF c_check_acode_type%NOTFOUND THEN
        /* did not find analysis code in table, raise exception */
        CLOSE c_check_acode_type;
        RAISE e_bad_ac_seq;
      END IF;
    CLOSE c_check_acode_type;
  END IF;

  IF v_dummy IS NULL AND p_row_pk.is_number IS NULL THEN
    /* kms - commented out below since the option to store characteristic default information in SIM
       was taken out of the current version of sim. Will now write error message instead. */
    /* used analysis code not mapped to a row, get info from sim table */
    /*kms    OPEN c_get_acode_defs(p_analysis_seq);
        FETCH c_get_acode_defs INTO v_acode_defs_rec.fraction, v_acode_defs_rec.uom_pk.is_number, v_acode_defs_rec.uom_pk.org_id,
      v_acode_defs_rec.dur_basis, v_acode_defs_rec.stat_type, v_acode_defs_rec.value_type, v_acode_defs_rec.temp_basis,
      v_acode_defs_rec.anlpr_pk.is_number, v_acode_defs_rec.anlpr_pk.org_id, v_acode_defs_rec.labprep_pk.is_number,
      v_acode_defs_rec.labprep_pk.org_id,v_acode_defs_rec.char_pk.is_number, v_acode_defs_rec.char_pk.org_id;
      IF c_get_acode_defs%NOTFOUND THEN*/
        /* did not find analysis code in table, raise exception */
     /*kms   CLOSE c_get_acode_defs;
        RAISE e_bad_ac_seq;
      END IF;
      CLOSE c_get_acode_defs;*/

    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Analysis code has not been mapped to a STORET group and row.',NULL,p_fa_seq,NULL);
  ELSE
    IF p_row_pk.is_number IS NOT NULL THEN
      /* kms - used characteristic group and row */
      OPEN c_get_storet_row_defs(p_row_pk);
      FETCH c_get_storet_row_defs INTO v_acode_defs_rec.fraction, v_uom_name, v_dur_basis, v_stat_type, v_value_type,
        v_temp_basis, v_acode_defs_rec.anlpr_pk.is_number, v_acode_defs_rec.anlpr_pk.org_id,
        v_acode_defs_rec.labprep_pk.is_number, v_acode_defs_rec.labprep_pk.org_id,v_acode_defs_rec.char_pk.is_number,
        v_acode_defs_rec.char_pk.org_id;
        IF c_get_storet_row_defs%NOTFOUND THEN
          /* kms - row primary key could not be found in STORET, raise exception */
          CLOSE c_get_storet_row_defs;
          RAISE e_bad_def_pk;
        END IF;
      CLOSE c_get_storet_row_defs;

      v_acode_defs_rec.lab_pk.is_number := NULL;
      v_acode_defs_rec.lab_pk.org_id := NULL;
    END IF;

    IF p_row_pk.is_number IS NULL THEN
      /* kms - used analysis code mapped to a row, get info from STORET tables */
      OPEN c_get_storet_defs(p_analysis_seq);
      FETCH c_get_storet_defs INTO v_acode_defs_rec.fraction, v_uom_name, v_dur_basis, v_stat_type, v_value_type,
        v_temp_basis, v_acode_defs_rec.anlpr_pk.is_number, v_acode_defs_rec.anlpr_pk.org_id,
        v_acode_defs_rec.labprep_pk.is_number, v_acode_defs_rec.labprep_pk.org_id,v_acode_defs_rec.char_pk.is_number,
        v_acode_defs_rec.char_pk.org_id,v_acode_defs_rec.lab_pk.is_number,v_acode_defs_rec.lab_pk.org_id;
        IF c_get_storet_defs%NOTFOUND THEN
          /* did not map analysis code to STORET tables, raise exception */
          CLOSE c_get_storet_defs;
          RAISE e_bad_def_map;
        END IF;
      CLOSE c_get_storet_defs;
    END IF;

    /* look up pk, allowable value sequences */
    /* get uom pk */
    v_char_result_type := get_char_result_type(v_acode_defs_rec.char_pk.is_number);
    IF NVL(v_char_result_type,'x') NOT IN ('TEXT','PV  ') THEN -- DCL added to remove UOM error generated even if there are no units of measure for a text or pv result type
     v_acode_defs_rec.uom_pk := get_uom_is_nbr(v_uom_name, p_fa_seq);
    END IF;

    /* get duration basis sequence */
    IF v_dur_basis IS NOT NULL THEN
      /* bka - need duration basis sequence in order to get allowable value */
      v_col_seq := get_col_seq('Duration Basis', p_imp_cfg_seq, p_fa_seq);

      v_acode_defs_rec.dur_basis := check_allowable_values(RTRIM(v_dur_basis), p_imp_cfg_seq, v_col_seq, p_fa_seq);

      v_col_seq := NULL;
    END IF;

    /* get stat type sequence */
    IF v_stat_type IS NOT NULL THEN
      /* bka - need statistic type sequence in order to get allowable value */
      v_col_seq := get_col_seq('Statistic Type', p_imp_cfg_seq, p_fa_seq);

      v_acode_defs_rec.stat_type := check_allowable_values(RTRIM(v_stat_type), p_imp_cfg_seq, v_col_seq, p_fa_seq);

      v_col_seq := NULL;
    END IF;

    /* get value type sequence */
    IF v_value_type IS NOT NULL THEN
      /* bka - need value type sequence in order to get allowable value */
      v_col_seq := get_col_seq('Value Type', p_imp_cfg_seq, p_fa_seq);

      v_acode_defs_rec.value_type := check_allowable_values(RTRIM(v_value_type), p_imp_cfg_seq, v_col_seq, p_fa_seq);

      v_col_seq := NULL;
   ELSE
      Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Value type has not been found and is necessary.',NULL,p_fa_seq,NULL);
    END IF;

    /* get temp basis sequence */
    IF v_temp_basis IS NOT NULL THEN
      /* bka - need temperature basis sequence in order to get allowable value */
      v_col_seq := get_col_seq('Temperature Basis', p_imp_cfg_seq, p_fa_seq);

      v_acode_defs_rec.temp_basis := check_allowable_values(RTRIM(v_temp_basis), p_imp_cfg_seq, v_col_seq, p_fa_seq);

      v_col_seq := NULL;
    END IF;

  END IF;

  RETURN 1;

EXCEPTION
  WHEN e_bad_ac_seq THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Unable to get analysis code.',NULL,p_fa_seq,NULL);
    v_acode_defs_rec := v_acode_copy_rec;
    RETURN 0;

  WHEN e_bad_def_map THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Unable to get char defaults from STORET due to faulty mapping.',NULL,p_fa_seq,NULL);
    v_acode_defs_rec := v_acode_copy_rec;
    RETURN 0;

  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in get char defaults.',NULL,p_fa_seq,NULL);
    v_acode_defs_rec := v_acode_copy_rec;
    RETURN 0;

END get_char_defaults;

/******************************************************************************************************/
FUNCTION get_value_msr (p_result_text IN VARCHAR2, p_result_ok OUT BOOLEAN, p_fa_seq IN NUMBER) RETURN NUMBER IS

  v_result_value       NUMBER(1);
  v_result_number      NUMBER;

BEGIN
  BEGIN
     /* see if the result is a number - translate result will be null if it is */
     IF TRANSLATE(p_result_text, 'a1234567890.- ', 'a') IS NULL THEN
     
       BEGIN -- bka - check again to see if the result value is 9.8.2 or 9-8 or 9..8
         v_result_number := TO_NUMBER(p_result_text);
                   
         p_result_ok := TRUE;
 
         /* see if the first part of the number is longer that 8 */
         IF INSTR(p_result_text,'.') > 9 THEN
           /* set flag to change activity status, keep the text value the same */
           v_result_value := 1;
           p_result_ok := FALSE;
           Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Result is too large for STORET: '||p_result_text,NULL,p_fa_seq,NULL);
         ELSIF INSTR(p_result_text,'.') = 0 THEN
           /* if the number does not have a decimal and is larger than 8, set set flag to change activity status */
           IF LENGTH(p_result_text) > 8 THEN
             v_result_value := 1;
             p_result_ok := FALSE;
             Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Result too large for STORET: '||p_result_text,NULL,p_fa_seq,NULL);
           END IF;
         ELSIF LENGTH(RTRIM(p_result_text)) - INSTR(p_result_text,'.') > 5 THEN
           /* if result tail is longer than 5 decimal places, warn user that it will be rounded */
           Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning: Result, '||p_result_text||' will be rounded to 5 places.',NULL,p_fa_seq,NULL);
         END IF;
       EXCEPTION WHEN OTHERS THEN
         v_result_value := 0;
         p_result_ok := FALSE;
         Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Result, '||p_result_text||', is not a number.', NULL,p_fa_seq,NULL);
       END;
      
     ELSE
       /* kms (5/29/01) - if the data in the result text field is not a number, set result_value
          to NULL and write error message */
       v_result_value := 0;
       p_result_ok := FALSE;
       Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Result, '||p_result_text||', is not a number.', NULL,p_fa_seq,NULL);
     END IF;

  EXCEPTION
    WHEN OTHERS THEN
      v_result_value := NULL;
      Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - checking result: '||v_result_value,NULL,p_fa_seq,NULL);
  END;

  RETURN v_result_value;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in get value measure.',NULL,p_fa_seq,NULL);
    RETURN NULL;

END get_value_msr;

/******************************************************************************************************/
FUNCTION validate_pv_result(p_orig_item IN VARCHAR2, p_imp_cfg_seq IN NUMBER,
                            p_char_is_nbr IN NUMBER, p_fa_seq IN NUMBER, p_ok OUT BOOLEAN) RETURN VARCHAR2 IS

  v_converted_item VARCHAR2(80);
  v_trans_exist    BOOLEAN;
  v_instr_code     VARCHAR2(15);

  v_dummy          NUMBER(8);
  v_result         SIM_RESULTS.RS_RESULT%TYPE;
  v_col_seq        NUMBER(10);

  e_no_col_seq     EXCEPTION;

  CURSOR c_get_result_col_seq IS
    SELECT SIC_SEQ FROM SIM_IMP_COLS WHERE SIC_NAME = 'Result Value'
    AND SIC_SID_SEQ = 10001;  /* dcl Temp Solution for mods made for biol. imps  */

  CURSOR c_check_char_pv(p_result IN VARCHAR2, p_char_is_nbr IN NUMBER) IS
    SELECT TSRCPV_IS_NUMBER FROM TSRCPV WHERE UPPER(SHORT_NAME) = UPPER(RPAD(p_result,12))
    AND TSRCHAR_IS_NUMBER = p_char_is_nbr;

BEGIN
  /* kms - function called if the result has a characteristic that indicates that it
     must be one of a list of allowable values.  Checks result to make sure it is has
     been checked for a translation. */
  /* kms (7/16/01) - added code that checks translations and will create them if the result
     value is not a permitted value and the instructions have been set to create translations. */

  /* kms (7/16/01) - get column sequence for results in order to see if any translations are
     set up for the result column. */
  p_ok := TRUE;
  
  OPEN c_get_result_col_seq;
  FETCH c_get_result_col_seq INTO v_col_seq;
    IF c_get_result_col_seq%NOTFOUND THEN
      CLOSE c_get_result_col_seq;
      RAISE e_no_col_seq;
    END IF;
  CLOSE c_get_result_col_seq;

 
  /* look in translations table to see if it needs to swapped */
  v_converted_item := check_translations(p_orig_item, p_imp_cfg_seq, v_col_seq, v_trans_exist);

  IF v_converted_item IS NOT NULL THEN

    /* check result to see if it is one of the permitted values */
    OPEN c_check_char_pv(v_converted_item, p_char_is_nbr);
    FETCH c_check_char_pv INTO v_dummy;
      IF c_check_char_pv%FOUND THEN
        v_result := p_orig_item;
      END IF;
    CLOSE c_check_char_pv;

    IF v_dummy IS NULL AND NOT v_trans_exist THEN
      /* check instructions to see if translation record should be created */
      v_instr_code := get_instructions(p_imp_cfg_seq, 'RES-TR', p_fa_seq);

      IF v_instr_code = 'RES-TR-01' THEN

        /* if record could not be matched to anything in translations or STORET tables,
           create new record */
        create_trans_record(p_orig_item, p_imp_cfg_seq, v_col_seq, p_fa_seq);

      END IF;

    ELSIF v_dummy IS NULL AND v_trans_exist THEN

      /*  item has been translated to an invalid storet value */
      Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_orig_item||' has been mapped to an invalid storet value: '||v_converted_item,NULL,p_fa_seq,NULL);
      --v_result := NULL;
      p_ok := FALSE;
    END IF;

  ELSIF v_trans_exist THEN
    /* item exists in translations table but is not mapped to anything, write error message.
       Status will be changed to 'N' later in the program. */
    --v_result := p_orig_item;
    --v_result := NULL;
    Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': '||p_orig_item||' exists in translation table but has not been mapped to a STORET code. ',NULL,p_fa_seq,NULL);
    p_ok := FALSE;
  END IF;

   RETURN v_result;

EXCEPTION
  WHEN e_no_col_seq THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Unable to find sequence for result column in function ''validate_pv_result''.',NULL,p_fa_seq,NULL);
    RETURN p_orig_item;

  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in validate pv_result.',NULL,p_fa_seq,NULL);
    RETURN p_orig_item;

END validate_pv_result;

/******************************************************************************************************/
FUNCTION update_result (p_result IN VARCHAR2, p_char_is_nbr IN NUMBER, p_units_ok OUT BOOLEAN,
                        p_result_ok OUT BOOLEAN, p_imp_cfg_seq IN NUMBER, p_fa_seq IN NUMBER)
                        RETURN VARCHAR2 IS


  v_result                     SIM_RESULTS.RS_RESULT%TYPE;
  v_success                    NUMBER(1);

/**  CURSOR c_get_char_type(p_char_is_nbr IN NUMBER) IS
    SELECT D_SCR_TYPE_CD FROM TSRCHAR
      WHERE TSRCHAR_IS_NUMBER = p_char_is_nbr
      AND TSRCHAR_ORG_ID = 'ZZZZZZZZ';
**/

BEGIN
  /* kms - function that gets characteristic type in order to see if it has permitted values
     or if other checks need to be performed on the result. */
/**
  OPEN c_get_char_type(p_char_is_nbr);
  FETCH c_get_char_type INTO v_char_type;
    IF c_get_char_type%NOTFOUND THEN
      v_char_type := NULL;
    END IF;
  CLOSE c_get_char_type;
**/


  /* kms (7/11/01) - reorganized code structure below in order to make it easier to read
     and to insure that get value msr is being called when appropriate. */
  IF v_char_result_type = 'PV  ' THEN
    /* kms (5/29/01) - result must be one of the permitted values for this characteristic. */
    -- v_char_result_type := 'PV  ';
    v_result := validate_pv_result(p_result,p_imp_cfg_seq,p_char_is_nbr,p_fa_seq,p_result_ok);
    p_units_ok := TRUE;

  ELSIF v_char_result_type = 'TEXT' THEN
    /* dcl (08/29/01) - result must be one of the permitted values for this characteristic. */
    v_result := '*Text';
   -- v_char_result_type := 'Text';
    p_units_ok := TRUE;

  ELSE
    p_units_ok := TRUE;
    v_result := p_result;

    IF v_char_result_type != 'PHYS' AND (p_result = '*Non-detect'
      OR p_result = '*Present <QL'
      OR p_result = '*Present >QL'
      OR p_result = '*Present') THEN
      /* kms (5/29/01) - result value not reported, units are not ok. **/
      v_result := p_result;
      p_units_ok := FALSE;
--sf - it actually turns out that PHYS characteristics can't have *Non-detect, etc.
    --ELSIF v_char_result_type = 'PHYS' AND (p_result = '*Non-detect'
      --OR p_result = '*Present <QL'
      --OR p_result = '*Present >QL'
      --OR p_result = '*Present') THEN
      /* kms (7/11/01) - if result has a physical characteristic, swap non-reported values with 0 */
      --  v_result := 0;

    ELSE

      /* kms (5/29/01) - result value was reported and is not for a characteristic with permitted
         values - the value in the result should be a number and units should be associated with
         the result if the characteristic calls for it. */

      /* kms (5/29/01) - check result text to make sure it is a number, write error if the value
         is too large, too precise, or not a number */

      /* DCL Result text does not need to be a number if it is a text char.  */
     IF v_char_result_type <> 'TEXT' THEN
      v_success := get_value_msr(v_result,p_result_ok,p_fa_seq);
     END IF;

    END IF;

  END IF;

  RETURN v_result;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in get char type. vres = '||v_result,NULL,p_fa_seq,NULL);
    RETURN NULL;

END update_result;

/******************************************************************************************************/
FUNCTION check_type_category_match(p_type_seq IN NUMBER, p_cat_seq IN NUMBER, p_cat_name OUT VARCHAR2,
                                   p_fa_seq IN NUMBER) RETURN NUMBER IS

  v_type_name                      SIM_IMP_ALLOW_VALUES.SIA_ITEM%TYPE;
  v_cat_name                       VARCHAR2(30);
  v_cat_ok                         NUMBER(1) := 0;

  e_bad_cat                        EXCEPTION;

  CURSOR c_get_allowable_value(p_sia_seq IN NUMBER) IS
    SELECT SIA_ITEM FROM SIM_IMP_ALLOW_VALUES WHERE SIA_SEQ = p_sia_seq;

BEGIN
  /* kms - function that makes sure category is valid for the activity type */

  /* get activity type */
  OPEN c_get_allowable_value(p_type_seq);
  FETCH c_get_allowable_value INTO v_type_name;
    IF c_get_allowable_value%NOTFOUND THEN
      v_type_name := NULL;
    END IF;
  CLOSE c_get_allowable_value;

  /* get activity category */
  OPEN c_get_allowable_value(p_cat_seq);
  FETCH c_get_allowable_value INTO v_cat_name;
    IF c_get_allowable_value%NOTFOUND THEN
      v_cat_name := NULL;
    END IF;
  CLOSE c_get_allowable_value;

  p_cat_name := v_cat_name;

  /* make sure category fits type */
  IF v_type_name = 'Sample' THEN
    /* set trip qc flag */
    v_trip_qc := 0;

    IF v_cat_name = 'Routine Sample'
      OR v_cat_name = 'Integrated Time Series'
      OR v_cat_name = 'Integrated Flow Proportioned'
      OR v_cat_name = 'Integrated Horizontal Profile'
      OR v_cat_name = 'Integrated Vertical Profile'
      OR v_cat_name = 'Composite w/o Parents'
      OR v_cat_name = 'Field Replicate'
      OR v_cat_name = 'Field Spike' THEN

      v_cat_ok := 1;
    END IF;

  ELSIF v_type_name = 'Field Msr/Obs' THEN
    /* set trip qc flag */
    v_trip_qc := 0;

    IF v_cat_name = 'Routine Msr/Obs'
      OR v_cat_name = 'Replicate Msr/Obs' THEN
      v_cat_ok := 1;
    END IF;

  ELSIF v_type_name = 'Trip QC' THEN
    /* set trip qc flag */
    v_trip_qc := 1;

    IF v_cat_name = 'Trip Blank'
      OR v_cat_name = 'Reagent Blank'
      OR v_cat_name = 'Equipment Blank'
      OR v_cat_name = 'Pre-preservative Blank'
      OR v_cat_name = 'Post-preservative Blank'
      OR v_cat_name = 'Field Spike'
      OR v_cat_name = 'Reference Sample' THEN
      v_cat_ok := 1;
    END IF;

  END IF;

  IF v_cat_ok != 1 THEN
    RAISE e_bad_cat;
  END IF;

  RETURN 1;

EXCEPTION
  WHEN e_bad_cat THEN
    RETURN 2;

  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in get check type category match.',NULL,p_fa_seq,NULL);
    RETURN 0;

END check_type_category_match;

/******************************************************************************************************/
FUNCTION check_uom_type (p_uom_is_nbr IN NUMBER, p_analysis_seq IN NUMBER, p_fa_seq IN NUMBER)
                         RETURN NUMBER IS

  v_uom_uom_type              TSRUOM.UOM_TYPE%TYPE;
  v_char_uom_type             TSRCHAR.UOM_TYPE%TYPE;

  e_wrong_char_type           EXCEPTION;

  CURSOR c_get_uom_type(p_uom_is_nbr IN NUMBER) IS
    SELECT UOM_TYPE FROM TSRUOM WHERE TSRUOM_IS_NUMBER = p_uom_is_nbr;

  CURSOR c_get_char_uom_type(p_analysis_seq IN NUMBER) IS
    SELECT UOM_TYPE FROM TSRCHAR, TSRCCGA, TSRCHDEF, SIM_ANALYSIS_CODES
      WHERE AC_TSRCHDEF_IS_NUMBER = TSRCHDEF_IS_NUMBER
      AND AC_TSRCHDEF_ORG_ID = TSRCHDEF_ORG_ID
      AND TSRCHDEF.TSRCCGA_IS_NUMBER = TSRCCGA.TSRCCGA_IS_NUMBER
      AND TSRCHDEF.TSRCCGA_ORG_ID = TSRCCGA.TSRCCGA_ORG_ID
      AND TSRCCGA.TSRCHAR_IS_NUMBER = TSRCHAR.TSRCHAR_IS_NUMBER
      AND TSRCCGA.TSRCHAR_ORG_ID = TSRCHAR.TSRCHAR_ORG_ID
      AND AC_SEQ = p_analysis_seq;

BEGIN
  /* get the uom type for the unit of measure and the characteristic */

  OPEN c_get_uom_type(p_uom_is_nbr);
  FETCH c_get_uom_type INTO v_uom_uom_type;
    IF c_get_uom_type%NOTFOUND THEN
      v_uom_uom_type := NULL;
    END IF;
  CLOSE c_get_uom_type;

  OPEN c_get_char_uom_type(p_analysis_seq);
  FETCH c_get_char_uom_type INTO v_char_uom_type;
    IF c_get_char_uom_type%NOTFOUND THEN
      v_char_uom_type := NULL;
    END IF;
  CLOSE c_get_char_uom_type;

  /* compare types, if they don't match, raise an exception that will write an error and change
     activity status to 'N' */
  IF RTRIM(v_uom_uom_type) != RTRIM(v_char_uom_type) THEN
    RAISE e_wrong_char_type;
  END IF;

  RETURN 1;

EXCEPTION
  WHEN e_wrong_char_type THEN
    RETURN 2;

  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - checking UOM types.',NULL,p_fa_seq,NULL);
    RETURN 0;

END check_uom_type;

/******************************************************************************************************/
FUNCTION validate_fraction (p_samp_frac_nm IN VARCHAR2, p_imp_cfg_seq IN NUMBER,
                            p_col_seq IN NUMBER, p_tsrchar_is_nbr IN NUMBER,
                            p_fa_seq IN NUMBER) RETURN VARCHAR2 IS

  v_sample_frac               VARCHAR2(40);
  v_samp_req_cd               VARCHAR2(1);

  v_trans_exist               BOOLEAN := TRUE;
  v_converted_item            VARCHAR2(80);
  v_instr_code                VARCHAR2(15);

  e_invalid_fraction          EXCEPTION;
  e_sample_frac_sp            EXCEPTION;

  CURSOR c_get_samp_req_cd (p_tsrchar_is_nbr IN NUMBER) IS
    SELECT SAMP_FRAC_REQ_CD FROM TSRCHAR WHERE tsrchar_is_number = p_tsrchar_is_nbr;

  CURSOR c_check_def_frac(p_fraction IN VARCHAR2) IS
    SELECT FIELD_VALUE FROM TSMPRMVL WHERE FIELD_NAME = RPAD('SMPL_FRAC_TYPE_NM',20)
    AND TABLE_NAME = RPAD('TSRRSULT',8) AND UPPER(FIELD_VALUE) = RPAD(UPPER(p_fraction),40);

BEGIN
  /* kms - validate sample fraction againsts values in STORET permitted values table. */
  /* look in translations table to see if it needs to swapped - will return original item or translation */
  v_converted_item := check_translations(p_samp_frac_nm, p_imp_cfg_seq, p_col_seq, v_trans_exist);

  IF v_converted_item IS NOT NULL OR p_samp_frac_nm IS NULL THEN

    /* set sample fraction to null if it is full of spaces */
    IF RPAD(v_converted_item,15) = RPAD(' ',15) OR p_samp_frac_nm IS NULL THEN
      v_sample_frac := NULL;
    ELSE
      v_sample_frac := v_converted_item;
    END IF;

    OPEN c_get_samp_req_cd(p_tsrchar_is_nbr);
    FETCH c_get_samp_req_cd INTO v_samp_req_cd;
      IF v_samp_req_cd = 'Y' OR v_sample_frac IS NOT NULL THEN

        /* see if fraction is valid if it is required or if it has been provided */
        OPEN c_check_def_frac(v_converted_item);
        FETCH c_check_def_frac INTO v_sample_frac;
          IF c_check_def_frac%NOTFOUND THEN
            /* if fraction could not be found in permitted values table, return null and write error */
            v_sample_frac := NULL;

            IF NOT v_trans_exist THEN
              /* check instructions to see if translation record should be created */
              v_instr_code := get_instructions(p_imp_cfg_seq, 'RES-TR', p_fa_seq);

              IF v_instr_code = 'RES-TR-01' THEN

                /* if record could not be matched to anything in translations or allowable values tables, create new record */
                create_trans_record(p_samp_frac_nm, p_imp_cfg_seq, p_col_seq, p_fa_seq);
                Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Sample Fraction, '||v_converted_item||' is not a valid sample fraction - translation record has been created.',NULL,p_fa_seq,NULL);

              ELSE
                Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Sample Fraction, '||v_converted_item||' is not a valid sample fraction.',NULL,p_fa_seq,NULL);
              END IF;

            ELSIF v_trans_exist THEN

              /* item has been translated to an invalid storet value */
              Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_samp_frac_nm||' has been mapped to the invalid storet value: '||v_converted_item,NULL,p_fa_seq,NULL);

            END IF;

            CLOSE c_check_def_frac;
--kms            RAISE e_invalid_fraction;
          ELSE

            /* make sure capitalization is correct */
         -- commented out sng 3/5 so can get superfund to work...
         --     IF RPAD(v_sample_frac,40) != RPAD(p_samp_frac_nm,40) THEN
         --     CLOSE c_check_def_frac;
         --     CLOSE c_get_samp_req_cd;
         --     RAISE e_sample_frac_sp;
         --     END IF;

            v_sample_frac := RTRIM(v_sample_frac);

          END IF;
        CLOSE c_check_def_frac;


      END IF;
    CLOSE c_get_samp_req_cd;

  ELSIF v_trans_exist AND v_converted_item IS NULL THEN
    /* if there is a code in the translations table but no value to translate to, write error message
       status will be changed to 'N' later in the program */
    v_sample_frac := NULL;
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||p_samp_frac_nm||' exists in translations table but has not been mapped to a STORET code.',NULL,p_fa_seq,NULL);

  END IF;

  RETURN v_sample_frac;

EXCEPTION
  WHEN e_invalid_fraction THEN
    Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, 'Line '||v_line_cnt||': Fraction, '||p_samp_frac_nm||', is not a valid STORET value.',NULL,p_fa_seq,NULL);
    RETURN NULL;

  WHEN e_sample_frac_sp THEN
     Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, 'Line '||v_line_cnt||': Check fraction capitalization: (STORET/USER)'||p_samp_frac_nm||'/'||v_sample_frac,NULL,p_fa_seq,NULL);
     RETURN v_sample_frac;

  WHEN OTHERS THEN
     Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in validate sample fraction.',NULL,p_fa_seq,NULL);
     RETURN NULL;

END validate_fraction;

/******************************************************************************************************/
FUNCTION check_duplicate_result(p_char_is_nbr IN NUMBER, p_fa_seq IN NUMBER, p_fraction IN VARCHAR2, p_det_limit IN VARCHAR2, p_uom_is_nbr IN NUMBER, p_anlpr_is_nbr in NUMBER, p_labprep_is_nbr IN NUMBER, p_dur_seq IN NUMBER, p_temp_seq IN NUMBER, p_prec_seq IN NUMBER, p_conf_seq IN NUMBER, p_stat_seq IN NUMBER) RETURN BOOLEAN IS
/* EPA is not clear on the actual definition for a duplicate result. We thought an easy and fairly thorough
solution would be to compare all the stuff stored in the SIM_RESULTS table. This list would be
Characteristic, fraction, detection limit, UOM, Analytic Procedure, LSPP, DUration, Temp, percision,
confidence level, and statistic type.
However, this package has reached it's size limit under Oracle 7, so the absolute max I could eek out
of this was to compare Fraction, detection limit, UOM, and analy proc. If the char valu is the same and
any of these is different it is NOT considered a duplicate.  (I left the more complete cursor here
in case we wwant to expand the criteria later.) SF 11/1/01
 /* CURSOR c_check_dup_result(p_char_is_nbr IN NUMBER, p_fa_seq IN NUMBER) IS
    SELECT rs_seq, rs_fraction, rs_detect_limit, rs_tsruom_is_nbr, rs_tsranlpr_is_number, 
    rs_tsrlspp_is_number, rs_durbasis_seq, rs_tempbasis_seq, rs_precision, rs_conflevel_seq,
    rs_stat_type_seq FROM SIM_RESULTS 
    WHERE RS_FA_SEQ = p_fa_seq AND RS_TSRCHAR_IS_NUMBER = p_char_is_nbr; */
    
  CURSOR c_check_dup_result(p_char_is_nbr IN NUMBER, p_fa_seq IN NUMBER) IS
    SELECT rs_seq, rs_fraction, rs_detect_limit, rs_tsruom_is_nbr, rs_tsranlpr_is_number
    FROM SIM_RESULTS 
    WHERE RS_FA_SEQ = p_fa_seq AND RS_TSRCHAR_IS_NUMBER = p_char_is_nbr;
    
  --CURSOR c_get_unit(p_result_seq IN NUMBER) IS
    --SELECT rs_tsruom_is_nbr FROM SIM_RESULTS WHERE RS_SEQ = p_result_seq; 

  v_result_unique BOOLEAN;
  v_unit_is_nbr   NUMBER(8);
  v_result_rec    c_check_dup_result%ROWTYPE;

BEGIN
  -- kms - function that looks for results that have the same characteristic on the same activity.
  
  OPEN c_check_dup_result(p_char_is_nbr, p_fa_seq);
  v_result_unique := TRUE;
  LOOP
    FETCH c_check_dup_result INTO v_result_rec;
      IF c_check_dup_result%NOTFOUND THEN
        EXIT;
      ELSE
      -- kms - org specific coding - take out when migration of CO legacy data is complete! 
      -- SF - Removed for general use 
      
      --IF v_org_is_nbr = 28 AND  p_char_is_nbr = 481 THEN
        -- kms - this will cause results that are water temp, deg F to be skipped if they are
          -- on an activity that already has temperature, but in deg C.  Know that results with
          -- deg C will be loaded before those with deg F since the data in the text file has been
          -- ordered by analysis code and water temp, deg C comes before water temp, deg F. 
        -- kms - make sure result that is already there really is in deg C units and is not a deg F dup 
        --OPEN c_get_unit (v_result_seq);
        --FETCH c_get_unit INTO v_unit_is_nbr;
        --CLOSE c_get_unit;

        -- kms - if the units for the record that already there are deg C, then skip current result 
        --IF v_unit_is_nbr = 200 THEN
          --v_skip_result := TRUE;
--          Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Water temperature, deg C is already on this activity, skipping result with water temperature, deg F.',NULL, p_fa_seq, NULL);
  --      END IF;

    --  ELSE  
        v_result_unique := FALSE;     
  
        IF NVL(v_char_result_type,'x') IN ('TEXT','PV  ') THEN
          v_result_unique := FALSE;
          --v_skip_result := TRUE;
          Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Result with this characteristic already exists on this activity.',NULL,p_fa_seq,NULL);
        ELSE
        
        --there are lots of these so check them one at a time 
        --If any of these differ then the result is unique

          IF NVL(v_result_rec.rs_fraction,'x') != NVL(p_fraction,'x') THEN
            v_result_unique := TRUE;
          END IF;
          IF NVL(v_result_rec.rs_detect_limit,-999) != NVL(p_det_limit,-999) THEN
            v_result_unique := TRUE;
          END IF;
          IF NVL(v_result_rec.rs_tsruom_is_nbr,-999) != NVL(p_uom_is_nbr,-999) THEN 
            v_result_unique := TRUE;
          END IF;
          IF NVL(v_result_rec.rs_tsranlpr_is_number,-999) != NVL(p_anlpr_is_nbr,-999) THEN
             v_result_unique := TRUE;
          END IF;
          --IF NVL(v_result_rec.rs_tsrlspp_is_number,-999) != NVL(p_labprep_is_nbr,-999) THEN
            --v_result_unique := TRUE;
--          END IF;        
  --        IF NVL(v_result_rec.rs_durbasis_seq,-999) != NVL(p_dur_seq,-999) THEN
    --        v_result_unique := TRUE;
      --    END IF;        
        --  IF NVL(v_result_rec.rs_tempbasis_seq,-999) != NVL(p_temp_seq,-999) THEN
          --  v_result_unique := TRUE;
--          END IF;        
          --IF NVL(v_result_rec.rs_precision,-999) != NVL(p_prec_seq,-999) THEN
            -- v_result_unique := TRUE;
          --END IF;       
          --IF NVL(v_result_rec.rs_conflevel_seq,-999) != NVL(p_conf_seq,-999) THEN
            -- v_result_unique := TRUE;
          --END IF;       
          --IF NVL(v_result_rec.rs_stat_type_seq,-999) != NVL(p_stat_seq,-999) THEN
            --v_result_unique := TRUE;
          --END IF;        
        
          IF v_result_unique = FALSE THEN
            Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Duplicate result with this characteristic exists on this activity.',NULL,p_fa_seq,NULL);
		EXIT;
          END IF;  
        END IF;
      END IF;
    END LOOP;
  CLOSE c_check_dup_result;

  RETURN v_result_unique;

EXCEPTION
  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, SQLERRM||' - in check duplicate result.',NULL,p_fa_seq,NULL);
    RETURN FALSE;

END check_duplicate_result;

/******************************************************************************************************/
FUNCTION create_result (p_result_rec IN SIM_RESULTS%ROWTYPE) RETURN NUMBER IS

BEGIN

  /* create result using values in result record */
  INSERT INTO SIM_RESULTS (
    RS_SEQ,
    RS_AC_SEQ,
    RS_FA_SEQ,
    RS_RESULT,
    RS_COMMENT,
    RS_DETECT_LIMIT,
    RS_IMPORT_LINE_NO,
    RS_VALUE_STATUS_SEQ,
    RS_LAB_BATCH_ID,
    RS_LAB_CERT,
    RS_ANALYSIS_DATE,
    RS_ANALYSIS_TIME,
    RS_DET_LIMIT_UOM_NBR,
    RS_DET_LIMIT_UOM_ORG,
    RS_DETECT_COMMENT,
    RS_UPPER_QUANT_LIMIT,
    RS_LOWER_QUANT_LIMIT,
    RS_TSRCHAR_IS_NUMBER,
    RS_TSRCHAR_ORG_ID,
    RS_TSRUOM_IS_NBR,
    RS_TSRUOM_ORG_ID,
    RS_FRACTION,
    RS_VALUE_TYPE_SEQ,
    RS_STAT_TYPE_SEQ,
    RS_TSRANLPR_IS_NUMBER,
    RS_TSRANLPR_ORG_ID,
    RS_TSRLSPP_IS_NUMBER,
    RS_TSRLSPP_ORG_ID,
    RS_DURBASIS_SEQ,
    RS_TEMPBASIS_SEQ,
    RS_PRECISION,
    RS_CONFLEVEL_SEQ,
    RS_TSRCHGRP_IS_NUMBER,
    RS_TSRCHGRP_ORG_ID,
    RS_TSRCHDEF_IS_NUMBER,
    RS_TSRCHDEF_ORG_ID,
    RS_TSRLAB_IS_NUMBER,
    RS_TSRLAB_ORG_ID)
  VALUES (
    p_result_rec.RS_SEQ,                               /* RS_SEQ */
    p_result_rec.RS_AC_SEQ,                            /* RS_AC_SEQ */
    p_result_rec.RS_FA_SEQ,                            /* RS_FA_SEQ */
    p_result_rec.RS_RESULT,                            /* RS_RESULT */
    p_result_rec.RS_COMMENT,                           /* RS_COMMENT */
    p_result_rec.RS_DETECT_LIMIT,                      /* RS_DETECT_LIMIT */
    p_result_rec.RS_IMPORT_LINE_NO,                    /* RS_IMPORT_LINE_NO */
    p_result_rec.RS_VALUE_STATUS_SEQ,                  /* RS_VALUE_STATUS_SEQ */
    p_result_rec.RS_LAB_BATCH_ID,                      /* RS_LAB_BATCH_ID */
    p_result_rec.RS_LAB_CERT,                          /* RS_LAB_CERT */
    p_result_rec.RS_ANALYSIS_DATE,                     /* RS_ANALYSIS_DATE */
    p_result_rec.RS_ANALYSIS_TIME,                     /* RS_ANALYSIS_TIME */
    p_result_rec.RS_DET_LIMIT_UOM_NBR,                 /* RS_DET_LIMIT_UOM_NBR */
    p_result_rec.RS_DET_LIMIT_UOM_ORG,                 /* RS_DET_LIMIT_UOM_ORG */
    p_result_rec.RS_DETECT_COMMENT,                    /* RS_DETECT_COMMENT */
    p_result_rec.RS_UPPER_QUANT_LIMIT,                 /* RS_UPPER_QUANT_LIMIT */
    p_result_rec.RS_LOWER_QUANT_LIMIT,                 /* RS_LOWER_QUANT_LIMIT */
    p_result_rec.RS_TSRCHAR_IS_NUMBER,                 /* RS_TSRCHAR_IS_NUMBER */
    p_result_rec.RS_TSRCHAR_ORG_ID,                    /* RS_TSRCHAR_ORG_ID */
    p_result_rec.RS_TSRUOM_IS_NBR,                     /* RS_TSRUOM_IS_NBR */
    p_result_rec.RS_TSRUOM_ORG_ID,                     /* RS_TSRUOM_ORG_ID */
    p_result_rec.RS_FRACTION,                          /* RS_FRACTION */
    p_result_rec.RS_VALUE_TYPE_SEQ,                    /* RS_VALUE_TYPE_SEQ */
    p_result_rec.RS_STAT_TYPE_SEQ,                     /* RS_STAT_TYPE_SEQ */
    p_result_rec.RS_TSRANLPR_IS_NUMBER,                /* RS_TSRANLPR_IS_NUMBER */
    p_result_rec.RS_TSRANLPR_ORG_ID,                   /* RS_TSRANLPR_ORG_ID */
    p_result_rec.RS_TSRLSPP_IS_NUMBER,                 /* RS_TSRLSPP_IS_NUMBER */
    p_result_rec.RS_TSRLSPP_ORG_ID,                    /* RS_TSRLSPP_ORG_ID */
    p_result_rec.RS_DURBASIS_SEQ,                      /* RS_DURBASIS_SEQ */
    p_result_rec.RS_TEMPBASIS_SEQ,                     /* RS_TEMPBASIS_SEQ */
    p_result_rec.RS_PRECISION,                         /* RS_PRECISION */
    p_result_rec.RS_CONFLEVEL_SEQ,                     /* RS_CONFLEVEL_SEQ */
    p_result_rec.RS_TSRCHGRP_IS_NUMBER,                /* RS_TSRCHGRP_IS_NUMBER */
    p_result_rec.RS_TSRCHGRP_ORG_ID,                   /* RS_TSRCHGRP_ORG_ID */
    p_result_rec.RS_TSRCHDEF_IS_NUMBER,                /* RS_TSRCHDEF_IS_NUMBER */
    p_result_rec.RS_TSRCHDEF_ORG_ID,                    /* RS_TSRCHDEF_ORG_ID */
    p_result_rec.RS_TSRLAB_IS_NUMBER,
    p_result_rec.RS_TSRLAB_ORG_ID);

  COMMIT;

  RETURN 1;

EXCEPTION

  WHEN OTHERS THEN
    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - inserting result.',NULL,p_result_rec.rs_fa_seq,NULL);
    RETURN 0;

END create_result;

/******************************************************************************************************/
PROCEDURE import_results (p_org_id IN VARCHAR2, p_import_log_seq IN NUMBER,
                          p_imp_cfg_seq IN NUMBER,p_test_count IN NUMBER) IS

  v_status              VARCHAR2(1);
  v_approved_by         VARCHAR2(20);
  v_approved_on         DATE;

  v_InLine              VARCHAR2(1000);
  v_bad_parse           BOOLEAN;
  v_line_table          Sim2_Import_Pkg.col_table_type;
  v_personnel_table     Sim2_Import_Pkg.col_table_type;
  v_persn_is_table      Sim2_Import_Result_Pkg.col_table_num_type;
  v_persn_org_table     Sim2_Import_Result_Pkg.col_table_type;
  v_project_table       Sim2_Import_Pkg.col_table_type;
  v_proj_is_table       Sim2_Import_Result_Pkg.col_table_num_type;
  v_proj_org_table      Sim2_Import_Result_Pkg.col_table_type;
  v_default_table       Sim2_Import_Result_Pkg.col_table_type;
  v_pos_table           Sim2_Import_Result_Pkg.col_table_num_type;
  v_trip_table          Sim2_Import_Result_Pkg.col_table_type;
  v_stvst_id_table      Sim2_Import_Result_Pkg.t_stvst_id_type;
  v_delimiter           VARCHAR2(1);
  v_num_cols            NUMBER(3);
  v_col_seq             NUMBER(10);
  v_dflt_pos            NUMBER(3);

  v_item_name           VARCHAR2(50);
  v_item                VARCHAR2(256);
  v_instr_code          VARCHAR2(15);
  v_comp                NUMBER(1);
  v_gear_type_ok        NUMBER(1);
  v_success             NUMBER(1);
  v_dummy_seq           NUMBER(10);
  v_storet_key          t_storet_pk_type;
  v_temp_key            t_storet_pk_type;
  v_dummy_key           t_storet_pk_type;

  v_fa_rec              SIM_FIELD_ACTIVITIES%ROWTYPE;
  v_fa_copy_rec         SIM_FIELD_ACTIVITIES%ROWTYPE;
  v_result_rec          SIM_RESULTS%ROWTYPE;
  v_result_copy_rec     SIM_RESULTS%ROWTYPE;
  v_trip_rec            SIM_TRIPS%ROWTYPE;
  v_trip_copy_rec       SIM_TRIPS%ROWTYPE;
  v_visit_rec           SIM_STATION_VISITS%ROWTYPE;
  v_visit_copy_rec      SIM_STATION_VISITS%ROWTYPE;

  v_type_seq                NUMBER(10);
  v_cat_seq                 NUMBER(10);
  v_reldepth_seq            NUMBER(10);
  v_field_set_name          VARCHAR2(30);
  v_field_set_nbr           VARCHAR2(10);
  v_lowq_limit              VARCHAR2(12);
  v_upq_limit               VARCHAR2(12);
  v_detect_limit            VARCHAR2(12);
  v_det_comment             VARCHAR2(254);
  v_det_uom_is_nbr          NUMBER(8);
  v_det_uom_org_id          VARCHAR2(8);
  v_result_cmt              VARCHAR2(1999) := NULL;
  v_result_status_seq       NUMBER(10);
  v_batch_id                VARCHAR2(12);
  v_lab_cert                VARCHAR2(1);
  v_stvst_cmt               VARCHAR2(240);
  v_updepth                 NUMBER(7,2);
  v_lowdepth                NUMBER(7,2);
  v_thermo_seq              NUMBER(10);
  v_halo_seq                NUMBER(10);
  v_pycno_seq               NUMBER(10);
  v_depmeasfrom_seq         NUMBER(10);
  v_depthzone_seq           NUMBER(10);
  v_depth_ref_unit_seq      NUMBER(10);

  v_last_activity_id    VARCHAR2(12) := '$$$$$$$$$$$$';
  v_activity_id         CHAR(12);
  v_new_activity        NUMBER(1);
  v_activity_seq        NUMBER(10);
  v_replicate_nbr       VARCHAR2(3);
  v_last_rep_nbr        VARCHAR2(3);
  v_last_station_id     VARCHAR2(15);
  v_station_id          VARCHAR2(15);
  v_last_trip_seq       NUMBER(10);
  v_last_visit_seq      NUMBER(10);
  v_station_visit       VARCHAR2(8);
  v_act_is_nbr          NUMBER(8);
  v_act_org_id          VARCHAR2(8);

  v_activity_cnt        NUMBER(10) := 0;
  v_error_cnt           NUMBER(10) := 0;
  v_complete_cnt        NUMBER(10) := 0;
  v_skipped_results_cnt NUMBER(10) := 0;
  v_result_cnt          NUMBER(10) := 0;
  v_import_desc         VARCHAR2(200);
  v_analysis_seq        NUMBER(10);
  v_medium_seq          NUMBER(10);
  v_last_med_seq        NUMBER(10);
  v_comment             VARCHAR2(1999);
  v_tsmpersn_is_nbr     NUMBER(8);
  v_tsmpersn_org_id     VARCHAR2(8);
  v_type                VARCHAR2(20);
  v_last_type           VARCHAR2(20);

  v_depth_ref_seq       NUMBER(10);
  v_depth_ref           VARCHAR2(30);
  v_depth_unit_seq      NUMBER(10);
  v_drange_seq          NUMBER(10);
  v_tsmstatn_is_nbr     NUMBER(8);
  v_tsmstatn_org_id     VARCHAR2(8);
  v_visit_start_date    DATE;
  v_visit_start_time    DATE;
  v_visit_end_date      DATE;
  v_visit_end_time      DATE;
  v_visit_seq           NUMBER(10);
  v_tsrstvst_is_nbr     NUMBER(8);
  v_tsrstvst_org_id     VARCHAR2(8);
  v_tsrlab_is_nbr       NUMBER(8);
  v_tsrlab_org_id       VARCHAR2(8);
  v_trip_id             VARCHAR2(15);
  v_trip_seq            NUMBER(10);
  v_trip_is_nbr         NUMBER(8);
  v_trip_org_id         VARCHAR2(8);
  v_trip_start_date     DATE;
  v_proj_is_nbr         NUMBER(8);
  v_proj_org_id         VARCHAR2(8);
  v_gear_is_nbr         NUMBER(8);
  v_gear_org_id         VARCHAR2(8);
  v_gearcfg_is_nbr      NUMBER(8);
  v_gearcfg_org_id      VARCHAR2(8);
  v_proc_is_nbr         NUMBER(8);
  v_proc_org_id         VARCHAR2(8);
  v_sample_pres_is_nbr  NUMBER(8);
  v_sample_pres_org_id  VARCHAR2(8);
  v_depth               NUMBER(7,2) := NULL;
  v_start_date          DATE;
  v_start_date_val      VARCHAR2(15);
  v_start_time          DATE;
  v_time_zone           NUMBER(10);
  v_end_date            DATE;
  v_end_time            DATE;

  v_analysis_date       DATE;
  v_analysis_time       DATE;
  v_analysis_code       VARCHAR2(15);
  v_result              VARCHAR2(256); -- DCL Changed from 20 (09/05/01)
  v_result_save         VARCHAR2(256); -- char may not be large enough

  v_dummy               BOOLEAN;
  v_units_ok            BOOLEAN;
  v_result_ok           BOOLEAN;
  v_tsrchgrp_is_nbr     NUMBER(8);
  v_tsrchgrp_org_id     VARCHAR2(8);
  v_row_id              VARCHAR2(80); -- dcl this was originally 15 changed to 80 to accomodate imports with funny row id's
  v_tsrchdef_is_nbr     NUMBER(8);
  v_tsrchdef_org_id     VARCHAR2(8);
  v_char_is_nbr         NUMBER(8);
  v_char_org_id         VARCHAR2(8);
  v_uom_is_nbr          NUMBER(8);
  v_uom_org_id          VARCHAR2(8);
  v_fraction            VARCHAR2(40);
  v_value_type          VARCHAR2(10);
  v_value_type_seq      NUMBER(10);
  v_stat_type_seq       NUMBER(10);
  v_anlpr_is_nbr        NUMBER(8);
  v_anlpr_org_id        VARCHAR2(8);
  v_labprep_is_nbr      NUMBER(8);
  v_labprep_org_id      VARCHAR2(8);
  v_duration_seq        NUMBER(10);
  v_temp_basis_seq      NUMBER(10);
  v_precision           VARCHAR2(12);
  v_tsrchdef_is_number  NUMBER(8);
  v_conflvl_seq         NUMBER(10);
  v_tsrchgrp_is_number  NUMBER(8);

  v_result_value_msr    NUMBER(1);
  v_first_lab_id        VARCHAR2(8);
  v_no_person           NUMBER(2);
  v_no_projects         NUMBER(2);

  v_create_act_id_cd    VARCHAR2(1);

  e_fileread            EXCEPTION;
  e_bad_org             EXCEPTION;
  e_no_cfg              EXCEPTION;
  e_e_bad_delimiter     EXCEPTION;
  e_bad_delimiter       EXCEPTION;
  --e_NO_DATA_FOUND       EXCEPTION;
  --e_project_is          EXCEPTION;


  CURSOR c_activity_seq IS SELECT SIM_FIELD_ACTIVITIES_SEQ.NEXTVAL FROM DUAL;

  CURSOR c_get_result_seq IS SELECT SIM_RESULTS_SEQ.NEXTVAL FROM DUAL;

  CURSOR c_get_next_visit_seq IS SELECT SIM_STATION_VISITS_SEQ.NEXTVAL FROM DUAL;

  CURSOR c_get_next_line(p_import_seq IN NUMBER) IS
    SELECT  SIPL_SEQ, SIPL_LINE_TEXT
      FROM  SIM_IMP_LINES
      WHERE SIPL_SIL_SEQ = p_import_seq
      AND   SIPL_LINE_TEXT IS NOT NULL   -- bka  added to eliminate null lines from the import.
      ORDER BY SIPL_LINE_NO;

  CURSOR c_column_loop(p_imp_cfg_seq IN NUMBER) IS
    SELECT SIC_PROC_ORDER, SIC_DFLT_POS, SIC_NAME, SIC_SEQ
      FROM SIM_IMP_COLS, SIM_IMP_DEFS, SIM_IMP_CFG
      WHERE SICC_SID_SEQ = SID_SEQ AND SIC_SID_SEQ = SID_SEQ AND SICC_SEQ = p_imp_cfg_seq
      ORDER BY SIC_PROC_ORDER;

  CURSOR c_comp_count(p_import_log_seq IN NUMBER) IS
    SELECT COUNT(RS_SEQ) FROM SIM_FIELD_ACTIVITIES, SIM_RESULTS
      WHERE RS_FA_SEQ = FA_SEQ AND FA_IMPORT_SEQ = p_import_log_seq
      AND FA_STATUS = 'A';

  CURSOR c_trip_dates (p_trip_seq IN NUMBER) IS
      SELECT STP_START_DATE FROM SIM_TRIPS WHERE STP_SEQ = p_trip_seq;

  --CURSOR c_create_act_id_cd IS
    --SELECT SC_VALUE FROM SIM_CONFIG WHERE SC_ITEM = 'CREATE_ACTIVITY_ID';

BEGIN
  /* kms - initialize global variables for this time through the package */
  v_line_cnt := 0;
  v_acode_defs_rec :=  v_acode_copy_rec;
  v_org_is_nbr := NULL;
  v_import_log_seq := 0;
  v_trip_qc := 0;
  v_cur_trip_is_nbr := NULL;
  --v_cur_proj_is_nbr := NULL;

  v_last_visit_id := 0;

  /* GET sequence for import log */
  v_import_log_seq := p_import_log_seq;

  BEGIN

    Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq, 'Start at '||TO_CHAR(SYSDATE,'HH24:MI:SS'),NULL,NULL,NULL);

    /* get tsmorgan_is_number */
    BEGIN
      v_org_is_nbr := get_org_info(p_org_id);
      IF v_success != 1 OR v_org_is_nbr IS NULL THEN
        RAISE e_bad_org;
      END IF;

    EXCEPTION
      WHEN e_bad_org THEN
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Unable to find the specified org id in local version of STORET.',NULL,NULL,NULL);
        RAISE NO_DATA_FOUND;

      WHEN OTHERS THEN
        Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq , SQLERRM||' - getting org is number.',NULL,NULL,NULL);
        RAISE NO_DATA_FOUND;
    END;

    /*  get the delimiter */
    BEGIN
      v_delimiter := get_delimiter(p_imp_cfg_seq);
      IF v_delimiter IS NULL THEN
        RAISE e_no_cfg;
      END IF;

    EXCEPTION
      WHEN e_no_cfg THEN
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Configuration invalid or does not exist - select new configuration or modify existing one.', NULL,NULL,NULL);
        RAISE NO_DATA_FOUND;

      WHEN OTHERS THEN
        Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq , SQLERRM||' - getting delimiter.',NULL,NULL,NULL);
        RAISE NO_DATA_FOUND;
    END;

    /* get actual positions and default information for items being imported */
    BEGIN

      v_success := get_item_pos(p_imp_cfg_seq, v_pos_table, v_default_table,v_num_cols);
      IF v_success != 1 THEN
        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Unable to find the import type in the import configurations - select or create a new configuration.',NULL,NULL,NULL);
      END IF;

    END;

    /* loop through the lines in the text table */
    FOR v_text_line IN c_get_next_line(v_import_log_seq) LOOP

      BEGIN
        /* read one line */
        v_inline := v_text_line.SIPL_LINE_TEXT;
        /** dcl 09/13/01 null values which might be carried through to the next line **/
        v_comment    := NULL;
        v_result_cmt := NULL;
        v_result     := NULL;
        v_line_cnt := v_line_cnt + 1;
        v_new_activity := 1; -- set to create new activity at beginning of loop
        v_status := 'A';
        --v_skip_result := FALSE;

        -- send line to parsing routine:
        BEGIN
          Sim2_Import_Pkg.PARSE_LINE_TABLE(v_inline, v_line_table, v_num_cols, v_delimiter,
                                           v_bad_parse,v_import_log_seq);

          IF v_bad_parse THEN
            RAISE e_e_bad_delimiter;
          END IF;

        EXCEPTION
          WHEN e_e_bad_delimiter THEN
            RAISE e_bad_delimiter;

          WHEN OTHERS THEN
            Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq , SQLERRM||' parsing one line.',NULL,NULL,NULL);
            RAISE e_fileread;
        END;

        /* loop through all result columns */
        FOR v_columns IN c_column_loop(p_imp_cfg_seq) LOOP

        /* get dflt position, item name, and column sequence */
        v_dflt_pos := v_columns.SIC_DFLT_POS;
        v_item_name := v_columns.SIC_NAME;
        v_col_seq := v_columns.SIC_SEQ;

        /* if column is included on the import, continue */
        IF v_pos_table(v_dflt_pos) IS NOT NULL THEN

          /* get the actual item, or use default if null */
          IF v_line_table(v_pos_table(v_dflt_pos)) IS NOT NULL THEN

--            v_item := v_line_table(v_pos_table(v_dflt_pos));
            IF LENGTH(v_line_table(v_pos_table(v_dflt_pos))) > 256 THEN
                 Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning: too many characters, truncating to 256',NULL,NULL, NULL);
            END IF;
            v_item := SUBSTR(v_line_table(v_pos_table(v_dflt_pos)),1,256);
          ELSIF v_line_table(v_pos_table(v_dflt_pos)) IS NULL AND v_default_table(v_dflt_pos) IS NOT NULL THEN
--            v_item := v_default_table(v_dflt_pos);
            IF LENGTH(v_line_table(v_pos_table(v_dflt_pos))) > 256 THEN
                 Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning: too many characters, truncating to 256',NULL,NULL, NULL);
            END IF;
            v_item := SUBSTR(v_default_table(v_dflt_pos),1,256);
          ELSE
            v_item := NULL;
          END IF;

          /* assign item to appropriate variables and take care of any other procedural issues that need to be
             taken care of for or with a particular item */
          IF v_item_name = 'Replicate Number' THEN
            v_replicate_nbr := v_item;

          ELSIF v_item_name = 'Activity Type' THEN
            /* kms (5/22/01) - activity type, category, station, and station visit will now all be processed
               before the activity id in order to insure that we are handling the uniqueness of activity ids
               correctly (must be unique on a station visit).  Activity type and category have been moved
               up in the processing order so those activities that are actually trip qcs do not get error
               messages concerning station ids or station visits - these are not included on trip qcs.  Any
               error messages concerning the validity of these items will be associated with the activity
               after the activity seq has been obtained.  This will be accomplished with an update statement
               that will assign all import detail records with 'Line: '||line_nbr in their description text
               to the appropriate activity. */
               
            --v_last_type := v_type;

            /* get the allowable value sequence for the activity type */
            v_type_seq := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,
                                              NULL);

            IF v_type_seq IS NULL THEN
              v_status := 'N';
              v_type := NULL;
            ELSE
              /* kms - need to keep value of activity type in order to make sure certain things are
                 done for samples only, i.e. sample collection procedure */
              v_type := v_item;
            END IF;

          ELSIF v_item_name = 'Activity Category' THEN
            IF v_item IS NOT NULL THEN

              /* get category - check translations and allowable values tables */
              v_cat_seq := validate_alval_item(v_item_name,v_item, p_imp_cfg_seq,v_col_seq,NULL);

              IF v_cat_seq IS NOT NULL THEN
                /* make sure valid for activity type */
                v_success := check_type_category_match(v_type_seq,v_cat_seq,v_item,NULL);

                IF v_success = 2 THEN
                  v_status := 'N';
                  Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Activity category is not valid for the chosen activity type.',NULL,NULL,NULL);
                END IF;

                /* replicate check */
                IF v_replicate_nbr IS NULL AND (v_item = 'Field Replicate' OR v_item = 'Replicate Msr/Obs') THEN

                  /* if activity indicates a replicate and it has been left blank, write error message */
                  v_status := 'N';
                  Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Activity category indicates the activity is a replicate, but no replicate number has been provided.',NULL,NULL,NULL);

                ELSIF v_replicate_nbr IS NOT NULL AND (v_item != 'Field Replicate' AND v_item != 'Replicate Msr/Obs') THEN

                  /* if activity is not a replicate and a replicate number has been given, write error message */
                  Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning: Activity is not a replicate, but a replicate number has been provided.', NULL,NULL, NULL);
                END IF;

              ELSE
                v_status := 'N';
              END IF;

            ELSE
              v_cat_seq := NULL;
              v_status := 'N';
              Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Activity category is required and has not been provided.', NULL,NULL,NULL);
            END IF;

          ELSIF v_item_name = 'Trip ID' THEN
            /* check instructions to see what the program should do */
            v_item       := UPPER(v_item); -- dcl all caps for county name
            v_instr_code := get_instructions(p_imp_cfg_seq, 'RES-TRIP', v_dummy_seq);

            IF v_instr_code IS NOT NULL THEN

              IF v_instr_code = 'RES-TRIP-04' THEN
                IF v_item IS NOT NULL THEN
                  /* kms (6/4/01) - set trip id equal to whatever is in the column (check for translation), will 
                                    validate later. */
                  v_trip_id := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy);
                ELSE
                  /* write error message and set status to 'N' if they have indicated that the trip id is in the data 
                     but it's not there */
                  Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Trip ID is required and has not been provided.  Either provide the trip id, or select an instruction choice to create trips.',NULL,v_dummy_seq,NULL);
                  v_status   := 'N';
                  v_trip_id  := NULL;
                  v_trip_seq := NULL;
                END IF;

              ELSE
                /* kms - get the trip id based on instructions */
                IF v_start_date IS NOT NULL THEN
                  v_trip_id := get_trip_id(v_instr_code, v_start_date, v_trip_rec.stp_start_date,
                                           v_trip_rec.stp_end_date, p_org_id, v_dummy_seq);
                ELSE
                  /* kms (7/11/01) - write error message explaining absence of trip when start date is not available. */
                  v_status   := 'N';
                  v_trip_seq := NULL;
                  v_trip_id  := NULL;
                  Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Activity Start Date is not available, Trip cannot be found or created.',
                                              NULL, v_dummy_seq, NULL);
                END IF;
              END IF;

              IF v_trip_id IS NOT NULL THEN
                /* kms (6/4/01) - check SIM table to see if trip is there, if not, check STORET and add the trip to 
                                  SIM. */
                v_trip_rec.stp_trip_id           := v_trip_id;
                v_trip_rec.stp_tsrtrip_is_number := NULL;
                v_trip_rec.stp_tsrtrip_org_id    := NULL;

                get_or_add_trip_rec(v_trip_rec, p_org_id, v_dummy_seq, v_trip_start_date);
               
                /* bka - if the trip has not been found or created, stop processing and flag v_status. */
                IF v_trip_rec.STP_SEQ IS NULL THEN
                  v_status   := 'N';
                  v_trip_seq := NULL;
                  v_trip_id  := NULL;
                ELSE
                  v_trip_seq             := v_trip_rec.stp_seq;
                  v_storet_key.is_number := v_trip_rec.stp_tsrtrip_is_number;
                  v_storet_key.org_id    := v_trip_rec.stp_tsrtrip_org_id;
  
                  /* kms (6/4/01) - if this is the first time the trip has been processed by this import
                     and it already exists in STORET, load all station visits on this trip in STORET
                     into the SIM table. */
                  IF v_storet_key.is_number IS NOT NULL AND NOT v_trip_table.EXISTS(v_trip_seq) THEN
                    load_station_visits(v_trip_seq, v_storet_key, p_org_id, v_stvst_id_table, v_dummy_seq);
                  END IF;
  
                  v_trip_table(v_trip_seq) := v_trip_id;
  
                  /* make sure activity falls within trip start and end dates */
                  IF v_trip_seq IS NOT NULL AND v_start_date IS NOT NULL THEN
                    v_success := check_trip_dates(v_trip_seq,v_start_date,p_org_id,v_dummy_seq);
  
                    IF v_success != 0 THEN
                      v_trip_is_nbr := v_storet_key.is_number;
                      v_trip_org_id := v_storet_key.org_id;
                    ELSE
                      v_trip_is_nbr := NULL;
                      v_trip_is_nbr := NULL;
                      v_trip_seq := NULL;
                      v_status := 'N';
                    END IF;
  
                  ELSIF v_trip_seq IS NOT NULL AND v_start_date IS NULL THEN
                    /* kms whenever the activity start date is null, set it to be the start date for the trip */
                    OPEN c_trip_dates(v_trip_seq);
                    FETCH c_trip_dates INTO v_start_date;
                    CLOSE c_trip_dates;
  
                    v_trip_is_nbr := v_storet_key.is_number;
                    v_trip_org_id := v_storet_key.org_id;
  
                    /* issue warning telling user the start date has been added */
                    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning: Activity Start Date has been set to '||v_start_date||', the trip start date.',
                                                NULL, v_dummy_seq, NULL);
                  ELSE
                    /* no trip, invalidate activity and make sure error message
                      written to the error log */
                    v_trip_is_nbr := NULL;
                    v_trip_is_nbr := NULL;
                    v_trip_seq    := NULL;
                    v_status      := 'N';
  
                    IF v_start_date IS NULL AND v_start_date_val IS NULL THEN
                      Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Start Date is required and has not been provided.',
                                                  NULL, v_dummy_seq, NULL);
                    END IF;
  
                  END IF;
                  
                END IF;

              END IF;

            ELSE
              /* if no instructions provided, write error message and set status to 'N' */
              v_trip_is_nbr := NULL;
              v_trip_org_id := NULL;
              v_trip_seq    := NULL;
              v_status      := 'N';
              Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': no instructions provided for trip, update current import configuration or select a new one.',
                                          NULL, NULL, NULL);
            END IF;

          ELSIF v_item_name = 'Trip Start Date' THEN
            IF v_item IS NOT NULL THEN
              v_trip_start_date := get_date(v_item_name, v_item, v_col_seq, p_imp_cfg_seq, v_activity_seq);
              IF v_trip_start_date IS NULL THEN
                v_status := 'N';
                Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Trip Start Date format not recognized.',
                                            NULL, NULL, NULL);
              END IF;
            ELSE
              v_trip_start_date := NULL;
            END IF;

          ELSIF v_item_name = 'Station ID' THEN
            IF v_item IS NOT NULL AND v_trip_qc != 1 THEN
              v_item := UPPER(v_item); -- Storet only does searches on station id's in upper case DCL
              v_station_id := v_item;

              /* get the station is number - check translations and STORET tables */
              v_storet_key := validate_storet_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_dummy_key,
                                                   p_org_id,v_line_cnt,v_dummy_seq);

              IF v_storet_key.is_number IS NOT NULL THEN
                v_tsmstatn_is_nbr := v_storet_key.is_number;
                v_tsmstatn_org_id := v_storet_key.org_id;
              ELSE
                /* station could not be found in storet, prevent export. */
                v_tsmstatn_is_nbr := NULL;
                v_tsmstatn_org_id := NULL;
                v_status := 'N';
              END IF;
            ELSIF v_trip_qc != 1 THEN
              v_tsmstatn_org_id := NULL;
              v_tsmstatn_is_nbr := NULL;
              v_status := 'N';
              Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Station ID is required and has not been provided.',NULL,NULL,NULL);
            ELSE
              v_tsmstatn_org_id := NULL;
              v_tsmstatn_is_nbr := NULL;
            END IF;

          ELSIF v_item_name = 'Station Visit Start Date' THEN
            v_start_date_val := v_item;

            IF v_item IS NOT NULL AND v_trip_qc != 1 THEN
              /* kms 6/2/01) - this date will be replaced later if the station visit already exists. */
              v_visit_start_date := get_date(v_item_name,v_item,v_col_seq,p_imp_cfg_seq,
                                             v_activity_seq);
              IF v_start_date IS NULL THEN
                v_status := 'N';
                Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Station Visit Start Date format not recognized.',NULL,NULL,NULL);
              END IF;
            ELSE
              /* kms - visit start date has not been included, wait to see if it can be obtained from
                 the station visit (if it already exists) or the trip. */
              v_visit_start_date := NULL;
            END IF;

          ELSIF v_item_name = 'Station Visit' THEN

            IF v_tsmstatn_is_nbr IS NOT NULL AND v_trip_qc != 1 THEN
              /* kms (5/22/01) - set temporary key to station key used to insure the
                 station visit is valid for this activity */
              v_temp_key.is_number := v_tsmstatn_is_nbr;
              v_temp_key.org_id := v_tsmstatn_org_id;
              v_storet_key.is_number := NULL;
              v_storet_key.org_id := NULL;

              /* check instructions to see what the program should do */
              v_instr_code := get_instructions(p_imp_cfg_seq,'RES-STVST',v_dummy_seq);

              IF v_instr_code IS NOT NULL THEN

                IF v_instr_code = 'RES-STVST-04' THEN
                  /* kms (5/31/01) - station visits included in import, station visit id
                     must there. */
                  IF v_item IS NOT NULL THEN

                    /* kms - get translated station visit number */
                    v_station_visit := check_translations(v_item, p_imp_cfg_seq, v_col_seq, v_dummy);

                    /* kms (5/29/01) - see if station visit exists in SIM_STATION_VISITS table */
                    v_visit_seq := get_visit_seq(v_storet_key, v_temp_key, v_station_visit,
                                                 v_trip_seq, v_dummy_seq);
                    /*                    
                    if v_visit_seq is not null then
                      insert into bka_test values('statn_is_number: '||v_tsmstatn_is_nbr||', ssv id: '||v_item||', start_date: '||to_char(v_visit_start_date,'MM/DD/RRRR'));
                    end if;
                    */
                  --  v_success := check_visit_start_date(v_tsmstatn_is_nbr,v_item,v_visit_start_date);
                  --  insert into bka_test values('v success: '||v_success);
                    /*
                    IF v_success != 1 THEN
                      Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': The station visit start date, '||v_visit_start_date||', provided in the import file does not match the station vistit start date found in STORET.',NULL,NULL,NULL);
                      v_status := 'N';
                    END IF;
                    */
                    /* kms - visit does not already exist in SIM or STORET and the start date has not
                       been included, get visit start date from the trip */
                    IF v_visit_seq IS NULL AND v_visit_start_date IS NULL THEN
                      v_visit_start_date := get_visit_start_date(v_storet_key,v_trip_seq,v_dummy_seq);
                    END IF;

                    /* kms - unable to get start date from the trip, write error message since visit
                       start date is required when the visit does not already exist and the start date
                       could not be obtained from the trip. */
                    IF v_visit_seq IS NULL AND v_visit_start_date IS NULL THEN
                      v_status := 'N';
                      Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Station Visit Start Date is required when the station visit does not already exist in STORET.  Unable to get start date from trip.',NULL,NULL,NULL);
                    END IF;

                  ELSE
                    /* kms (5/31/01) - station visit required, write error */
                    v_status := 'N';
                    v_visit_seq := NULL;
                    v_visit_start_date := NULL;
                    v_storet_key.is_number := NULL;
                    v_storet_key.org_id := NULL;
                    Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Station Visit is required when the instruction to read visits from the import file has been chosen.',NULL,NULL,NULL);
                  END IF;
                ELSE
                  /* kms (6/1/01) - creating station visits.  Get the max station visit id nbr
                     and load into table that holds current station visit id nbr indexed by trip seq. */

                  /* kms - get visit id based on instructions and get visit seq if it's in SIM. */
                  check_existing_visit(v_instr_code,v_start_date,v_start_time,v_trip_seq,v_temp_key,
                                       v_visit_start_date,v_visit_start_time,v_visit_end_date,
                                       v_visit_end_time,v_storet_key,v_visit_seq,v_stvst_id_table,
                                       v_station_visit);
                END IF;

                /* kms (6/2/01) - if station visit is not in the sim table, add */
                IF v_visit_seq IS NULL AND v_trip_seq IS NOT NULL THEN

                  /* kms - get next visit seq */
                  OPEN c_get_next_visit_seq;
                  FETCH c_get_next_visit_seq INTO v_visit_rec.SSV_SEQ;
                  CLOSE c_get_next_visit_seq;

                  /* kms - populate remaining columns in visit record. */
                  v_visit_rec.SSV_STP_SEQ := v_trip_seq;
                  v_visit_rec.SSV_ID := v_station_visit;
                  v_visit_rec.SSV_START_DATE := v_visit_start_date;
                  v_visit_rec.SSV_START_TIME := v_visit_start_time;
                  v_visit_rec.SSV_END_DATE := v_visit_end_date;
                  v_visit_rec.SSV_END_TIME := v_visit_end_time;
                  v_visit_rec.SSV_TSMSTATN_IS_NUMBER := v_tsmstatn_is_nbr;
                  v_visit_rec.SSV_TSMSTATN_ORG_ID := v_tsmstatn_org_id;
                  v_visit_rec.SSV_TSRSTVST_IS_NUMBER := v_storet_key.is_number;
                  v_visit_rec.SSV_TSRSTVST_ORG_ID := v_storet_key.org_id;
                  v_visit_rec.SSV_TSMORGAN_IS_NUMBER := v_org_is_nbr;
                  v_visit_rec.SSV_TSMORGAN_ORG_ID := p_org_id;
                  v_visit_rec.SSV_CREATED_BY_SIM := 1;

                  /* kms - create station visit */
                  v_success := create_sim_station_visit(v_visit_rec);

                  v_visit_seq := v_visit_rec.SSV_SEQ;
                ELSIF v_visit_seq IS NULL THEN
                  v_status := 'N';
                  Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Station Visit could not be added since there is no trip on this activity.',NULL,NULL,NULL);
                END IF;

                v_tsrstvst_is_nbr := v_storet_key.is_number;
                v_tsrstvst_org_id := v_storet_key.org_id;
                v_temp_key.is_number := NULL;
                v_temp_key.org_id := NULL;

              ELSE
                /* if no instructions provided, write error message and set status to 'N' */
                v_tsrstvst_is_nbr := NULL;
                v_tsrstvst_org_id := NULL;
                v_temp_key.is_number := NULL;
                v_temp_key.org_id := NULL;
                v_status := 'N';
                Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': no instructions provided for station visit, update current import configuration or select a new one.',NULL,NULL,NULL);
              END IF;

              /* kms (6/6/01)- make sure station visit start date comes before or is the same as
                 activity start date */
              IF v_visit_seq IS NOT NULL THEN
                v_success := check_visit_dates(v_visit_seq, v_start_date, p_org_id, v_dummy_seq);

                IF v_success != 1 THEN
                  v_visit_seq := NULL;
                  v_tsrstvst_is_nbr := NULL;
                  v_tsrstvst_org_id := NULL;
                  v_status:= 'N';
                END IF;
              END IF;
            ELSIF v_trip_qc != 1 THEN
              /* kms (6/25/01) - no station on the activity, so don't do anything with the station visit.
                 Write error message and invalidate the activity. */
              v_status := 'N';
              v_visit_seq := NULL;
              v_tsrstvst_is_nbr := NULL;
              v_tsrstvst_org_id := NULL;
              Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Unable to load Station Visit on this activity since Station was either not provided or is invalid.',NULL,v_dummy_seq,NULL);
            ELSE
              v_visit_seq := NULL;
              v_tsrstvst_is_nbr := NULL;
              v_tsrstvst_org_id := NULL;
            END IF;

          ELSIF v_item_name = 'Activity ID' THEN
            BEGIN
              IF v_item IS NOT NULL THEN
                IF length(v_item) > 12 THEN
                  v_activity_id := UPPER(RPAD(SUBSTR(v_item,1,12),12)); -- truncate to allowed length
                  Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning: Activity ID is longer that 12 characters and will be truncated.',NULL,v_activity_seq,NULL);
                ELSE
                  v_activity_id := UPPER(v_item); -- dcl id's need to always be uppers
                END IF;
              ELSE
                Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Activity ID is required and has not been provided.',NULL,v_activity_seq,NULL);
                v_activity_id := '????????';
                v_status := 'N';
              END IF;                
              /* kms added for Colorado legacy migration - remove for distribution or change
                 to hidden instruction */
              /* kms check configuration to make sure activity id should be created based on
                 value in the activity id field.  Default will be no. */

/*              OPEN c_create_act_id_cd;
              FETCH c_create_act_id_cd INTO v_create_act_id_cd;
                IF c_create_act_id_cd%NOTFOUND THEN
                  v_create_act_id_cd := 'N';
                END IF;
              CLOSE c_create_act_id_cd;

              -- kms set up to create activity ids, call function that will do this based on
                 value in activity id field 
              IF v_create_act_id_cd = 'Y' THEN
                v_activity_id := get_activity_id(v_item, p_org_id);
              END IF;
*/
              --  initialize counters and comparision variables
                IF v_line_cnt = 1 THEN

                -- first line
                  v_last_activity_id := v_activity_id;
                  v_last_rep_nbr := v_replicate_nbr;
                  v_last_trip_seq := v_trip_seq;
                  v_last_visit_seq := v_visit_seq;
                  v_last_med_seq := v_medium_seq;
                  v_last_type := v_type;
                  v_activity_cnt := 1;
                  v_new_activity := 1;

                ELSIF  v_activity_id != v_last_activity_id
                  OR NVL(v_replicate_nbr,'`') != NVL(v_last_rep_nbr,'`')
                  OR NVL(v_visit_seq,'-1') != NVL(v_last_visit_seq,'-1')
                  OR NVL(v_trip_seq,'-1') != NVL(v_last_trip_seq,'-1') THEN

                /* check for activity in sim_field_activities and decide if it needs to be made */
                  v_activity_seq := check_activity_id(v_activity_id,v_replicate_nbr,v_tsmstatn_is_nbr,
                                                    v_trip_seq,v_visit_seq,RPAD(p_org_id,8),
                                                    v_type_seq,v_medium_seq,v_status);
         
                  IF v_activity_seq IS NULL THEN

                  /* not first line, but new activity id */
                    v_last_activity_id := v_activity_id;
                    v_last_rep_nbr := v_replicate_nbr;
                    v_last_visit_seq := v_visit_seq;
                    v_last_trip_seq := v_trip_seq;
                    v_activity_cnt := v_activity_cnt + 1;
                    v_last_med_seq := v_medium_seq;
                    v_last_type := v_type;
                    v_new_activity := 1;
                  ELSE
 
                  /* activity already exists, do not create */
                    v_new_activity := 0;
                    v_last_activity_id := v_activity_id;
                    v_last_rep_nbr := v_replicate_nbr;
                    v_last_visit_seq := v_visit_seq;
                    v_last_trip_seq := v_trip_seq;
                    v_last_med_seq := v_medium_seq;
                    v_last_type := v_type;                  

                  END IF;

                ELSE
                /* Check the activity type and medium to make sure they are the same */

                  IF v_last_med_seq != v_medium_seq THEN
                    v_status := 'N';
                    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': A new medium for this activity has been found.  Only one medium per activity is allowed.',NULL,v_activity_seq,NULL);
                  ELSIF v_last_type != v_type THEN
                    v_status := 'N';
                    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': A new activity type for this activity has been found.  Only one activity type per activity is allowed.',NULL,v_activity_seq,NULL);
                  END IF;
                
                -- another line, same activity
                  v_new_activity := 0;
                  v_last_activity_id := v_activity_id;
                  v_last_rep_nbr := v_replicate_nbr;
                  v_last_visit_seq := v_visit_seq;
                  v_last_trip_seq := v_trip_seq;
                  v_last_med_seq := v_medium_seq;
                  v_last_type := v_type; 
                END IF;

                IF v_new_activity = 1 THEN
                -- Get sequence for field activity record (save so can use for related results)
                  BEGIN
                    OPEN c_activity_seq;
                    FETCH c_activity_seq INTO v_activity_seq;
                      IF c_activity_seq%NOTFOUND THEN
                        v_activity_seq := NULL;
                        Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Failed to get new activity sequence.',NULL,v_activity_seq,NULL);
                      END IF;
                    CLOSE c_activity_seq;

                  EXCEPTION
                    WHEN OTHERS THEN
                      Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||'- getting new activity sequence.',NULL,v_activity_seq,NULL);
                  END;
                END IF;

              /* kms - update error messages generated for this activity before activity sequence was obtained.
                 Use line nbr to find the error messages that need to be updated */
                BEGIN
                  UPDATE SIM_IMPORT_DETAILS
                    SET sid_fa_seq = v_activity_seq
                    WHERE sid_desc LIKE '%Line '||v_line_cnt||':%'
                    AND sid_sil_seq = v_import_log_seq;
                EXCEPTION
                  WHEN OTHERS THEN
                    Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||'- updating import errors.',NULL,v_activity_seq,NULL);
                END;
 
              EXCEPTION
                WHEN OTHERS THEN
                  v_import_desc := 'Line '||v_line_cnt||': '||SQLERRM||' - looking for pre-existing activity.';
                  Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, v_import_desc,NULL,v_activity_seq,NULL);

              END;

          ELSIF v_item_name = 'Visit Comments' AND v_new_activity = 1 THEN
            IF v_trip_qc != 1 THEN
              v_stvst_cmt := v_item;
            ELSE
              v_stvst_cmt := NULL;
            END IF;

          ELSIF v_item_name = 'Medium'  THEN  --need medium on every result to check char group id
            --v_last_med_seq := v_medium_seq;

            IF v_item IS NOT NULL THEN
              /* get medium - check translations and allowable values tables */
              v_medium_seq := validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq);

              IF v_medium_seq IS NULL THEN
                v_status := 'N';
              END IF;

            ELSE
              v_medium_seq := NULL;
              v_status := 'N';
              Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Medium is required and has not been provided.',NULL,v_activity_seq, NULL);
            END IF;

          ELSIF v_item_name = 'Activity Start Date' AND v_new_activity = 1 THEN
            BEGIN
              /* kms If start date is null, it will be set to the start date for the
                 activity's trip.  The following lines that change activity status were
                 commented in order to avoid invalidating the activity. Also commented out
                 lines to prevent repetitive error messages. */

              /* if there is no start date, set status to 'N' so it will not be exported */
              IF v_item IS NOT NULL THEN
                /* get start date using formatting options */
                v_start_date := get_date(v_item_name,v_item,v_col_seq,p_imp_cfg_seq,v_activity_seq);

                IF v_start_date IS NULL THEN
                  v_status := 'N';
                  v_start_date_val := v_item;
                END IF;
              ELSE
          --kms v_status := 'N';
                v_start_date := NULL;
          --kms sim2_import_pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Start Date is required and has not been provided.',null,v_activity_seq,null);
              END IF;

            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                v_start_date := NULL;
                v_start_time := NULL;
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - getting start date, line: '||v_line_cnt,NULL,v_activity_seq,NULL);
            END;

          ELSIF v_item_name = 'Activity Start Time' AND v_new_activity = 1 THEN
            BEGIN
              IF v_start_date IS NOT NULL THEN

                /* if a start time is supplied, format it for STORET */
                IF v_item IS NOT NULL /* AND v_trip_qc != 1 */ THEN

                  /* get start time using formatting options */
                  v_start_time := get_time(v_item_name,v_item,v_col_seq,p_imp_cfg_seq,v_activity_seq);

                  IF v_start_time IS NULL THEN
                    v_status := 'N';
                  END IF;

                ELSE
                  v_start_time := NULL;

                END IF;

              END IF;
            EXCEPTION
              WHEN OTHERS THEN
                v_start_time :=  NULL;
                v_start_date := NULL;
                v_status := 'N';
                v_import_desc := 'Line '||v_line_cnt||': Error in date/time "'||v_start_date||'/'||v_item||'".';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, v_import_desc,NULL,v_activity_seq,NULL);

            END;

          ELSIF v_item_name = 'Activity Stop Date' AND v_new_activity = 1 THEN
            BEGIN
              IF v_item IS NOT NULL AND v_trip_qc != 1 THEN
                v_end_date := get_date(v_item_name,v_item,v_col_seq,p_imp_cfg_seq,v_activity_seq);

                IF v_end_date IS NULL THEN
                  v_status := 'N';
                END IF;

              ELSE
                v_end_date := NULL;
              END IF;

            EXCEPTION
              WHEN OTHERS THEN
                v_status := 'N';
                v_end_date := NULL;
                v_end_time := NULL;
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - getting stop date.',NULL,v_activity_seq,NULL);
            END;

          ELSIF v_item_name = 'Activity Stop Time' AND v_new_activity = 1 THEN
            BEGIN
              IF v_end_date IS NOT NULL THEN
                IF v_item IS NOT NULL THEN

                  v_end_time := get_time(v_item_name,v_item,v_col_seq,p_imp_cfg_seq,v_activity_seq);

                  IF v_end_time IS NULL THEN
                    v_status := 'N';
                  END IF;

                ELSE
                  v_end_time := NULL;
                END IF;
              ELSE
                v_end_time := NULL;
              END IF;

            EXCEPTION
              WHEN OTHERS THEN
                v_end_time :=  NULL;
                v_end_date := NULL;
                v_import_desc := 'Line '||v_line_cnt||': Error in stop date/time "'||v_end_date||'/'||v_item||'".';
                v_status := 'N';
                Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, v_import_desc,NULL,v_activity_seq,NULL);
            END;

          ELSIF v_item_name = 'Start/Stop Time Zone' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              /* get allowable value sequence for time zone */
              v_time_zone := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,
                                                 v_activity_seq);

            ELSE
              v_time_zone := NULL;
            END IF;

          ELSIF v_item_name = 'Activity Comments' AND v_new_activity = 1 THEN
            IF v_trip_qc != 1 THEN
              /* comments */
              v_comment := RTRIM(v_item);
            ELSE
              v_comment := NULL;
            END IF;

          ELSIF v_item_name = 'Activity Depth' AND v_new_activity = 1 THEN
            IF v_trip_qc != 1 THEN
              v_depth := v_item;
            ELSE
              v_depth := NULL;
            END IF;

          ELSIF v_item_name = 'Depth Units' AND v_new_activity = 1 THEN

            IF v_item IS NOT NULL AND v_depth IS NOT NULL AND v_trip_qc != 1 THEN

              v_depth_unit_seq := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq);

            ELSIF v_depth IS NOT NULL AND v_trip_qc != 1  THEN
              v_depth_unit_seq := NULL;
              v_status := 'N';
              Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Depth Unit is required if depth has been provided.',NULL,v_activity_seq,NULL);
            ELSE
              v_depth_unit_seq := NULL;
            END IF;

          ELSIF v_item_name = 'Relative Depth' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL AND v_trip_qc != 1 THEN
              v_reldepth_seq := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq);
            ELSE
              v_reldepth_seq := NULL;
            END IF;

          ELSIF v_item_name = 'Upper Depth' AND v_new_activity = 1 THEN
            IF (v_depth IS NOT NULL OR v_reldepth_seq IS NOT NULL OR v_depth_unit_seq IS NOT NULL) AND v_item IS NOT NULL THEN
              v_status := 'N';
              Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Activity depth and/or related information has already been provided.  Both activity depth and upper/lower depth data are not allowed in STORET.',NULL, v_activity_seq, NULL);
              v_updepth := NULL;
            ELSE
              IF v_trip_qc != 1 THEN
                v_updepth := v_item;
              ELSE
                v_updepth := NULL;
              END IF;
            END IF;

          ELSIF v_item_name = 'Lower Depth' AND v_new_activity = 1 THEN
            IF (v_depth IS NOT NULL OR v_reldepth_seq IS NOT NULL OR v_depth_unit_seq IS NOT NULL) AND v_item IS NOT NULL THEN
              v_status := 'N';
              Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Activity depth and/or related information has already been provided.  Both activity depth and upper/lower depth data are not allowed in STORET.',NULL, v_activity_seq, NULL);
              v_lowdepth := NULL;
            ELSE
              IF v_trip_qc != 1 THEN
                v_lowdepth := v_item;
              ELSE
                v_lowdepth := NULL;
              END IF;
            END IF;

          ELSIF v_item_name = 'Upper/Lower Depth Units' AND v_new_activity = 1 THEN
            
            IF (v_depth IS NOT NULL OR v_reldepth_seq IS NOT NULL OR v_depth_unit_seq IS NOT NULL) AND v_item IS NOT NULL THEN
              v_status := 'N';
              Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Activity depth and/or related information has already been provided.  Both activity depth and upper/lower depth data are not allowed in STORET.',NULL, v_activity_seq, NULL);
              v_drange_seq := NULL;
            ELSE
              IF v_item IS NOT NULL AND v_lowdepth IS NOT NULL AND v_updepth IS NOT NULL AND v_trip_qc != 1 THEN
                /* get the depth range units allowable value - check translations table */
                v_drange_seq := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq);
  
                IF v_drange_seq IS NULL THEN
                  /* if unit of measure could not be found then set the status of the activity to N */
                  v_status := 'N';
                END IF;
  
              ELSE
                v_drange_seq := NULL;
              END IF;
            END IF;

          ELSIF v_item_name = 'Depth Measured From' AND v_new_activity = 1 THEN
            IF v_trip_qc != 1 THEN
              v_depth_ref := UPPER(v_item);
            ELSE
              v_depth_ref := NULL;
            END IF;

          ELSIF v_item_name = 'Depth Zone Type' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL AND v_trip_qc != 1 THEN

              v_depthzone_seq := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq);

              IF v_depthzone_seq IS NULL THEN
                v_status := 'N';
              END IF;

            ELSE
              v_depthzone_seq := NULL;
            END IF;

          ELSIF v_item_name = 'Thermocline' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL AND v_trip_qc != 1 THEN

              v_thermo_seq := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq);

              IF v_thermo_seq IS NULL THEN
                v_status := 'N';
              END IF;

            ELSE
              v_thermo_seq := NULL;
            END IF;

          ELSIF v_item_name = 'Halocline' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL AND v_trip_qc != 1 THEN

              v_halo_seq := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq);

              IF v_halo_seq IS NULL THEN
                v_status := 'N';
              END IF;

            ELSE
              v_halo_seq := NULL;
            END IF;

          ELSIF v_item_name = 'Pycnocline' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL AND v_trip_qc != 1 THEN

              v_pycno_seq := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq);

              IF v_pycno_seq IS NULL THEN
                v_status := 'N';
              END IF;

            ELSE
              v_pycno_seq := NULL;
            END IF;

          ELSIF v_item_name = 'Personnel' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              v_no_person := 0;

              /* can have multiple personnel separated by '\' - get number of personnel to expect */
              /* kms - translate function removes all occurences of the delimiter.  By subtracting
                 the length of what it returns from the length of the original, we find the number
                 of delimiters that were in the original.  We then add 1 since we are counting the
                 number of personnel, not the number of delimiters. */
              v_no_person := LENGTH(v_item) - LENGTH(TRANSLATE(v_item,'a\','a')) + 1;

              /* separate item into personnel using parse procedure if more than 1 person */
              IF v_no_person > 1 THEN
                BEGIN
                  Sim2_Import_Pkg.parse_line_table(v_item,v_personnel_table,v_no_person,'\',v_bad_parse,v_import_log_seq);
                EXCEPTION
                  WHEN NO_DATA_FOUND THEN
                    NULL;
                END;
              ELSE
                v_personnel_table(1) := v_item;
              END IF;

              /* loop through personnel table and add each person in column to the activity */
              FOR v_number IN 1..v_no_person LOOP

                v_storet_key := validate_storet_item(v_item_name,v_personnel_table(v_number),p_imp_cfg_seq,v_col_seq,
                                                     v_dummy_key,p_org_id,v_line_cnt,v_activity_seq);

                IF v_storet_key.is_number IS NULL THEN
                  v_persn_is_table(v_number) := NULL;
                  v_persn_org_table(v_number) := NULL;
                  v_status := 'N';
                ELSE
                  v_persn_is_table(v_number) := v_storet_key.is_number;
                  v_persn_org_table(v_number) := v_storet_key.org_id;
                END IF;

              END LOOP;

            ELSE
              v_no_person := 0;
              v_tsmpersn_is_nbr := NULL;
              v_tsmpersn_org_id := NULL;
            END IF;

          ELSIF v_item_name = 'Sample Collection Procedure' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL AND v_trip_qc != 1 THEN

              /* get the procedure is number - check translations and STORET tables */
              v_storet_key := validate_storet_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_dummy_key,
                                                   p_org_id,v_line_cnt,v_activity_seq);

              IF v_storet_key.is_number IS NOT NULL THEN
                v_proc_is_nbr := v_storet_key.is_number;
                v_proc_org_id := v_storet_key.org_id;
              ELSE
                v_proc_org_id := NULL;
                v_proc_is_nbr := NULL;
                v_status := 'N';
              END IF;

            ELSE
              v_proc_is_nbr := NULL;
              v_proc_org_id := NULL;
            END IF;

          ELSIF v_item_name = 'Gear ID' AND v_new_activity = 1 THEN
            v_item := UPPER(v_item); -- dcl id's need TO always be uppers
            IF v_item IS NOT NULL AND v_trip_qc != 1 THEN

              /* get the gear is number - check translations and STORET tables */
              v_storet_key := validate_storet_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_dummy_key,
                                                   p_org_id,v_line_cnt,v_activity_seq);

              IF v_storet_key.is_number IS NOT NULL THEN
                v_gear_is_nbr := v_storet_key.is_number;
                v_gear_org_id := v_storet_key.org_id;
              ELSE
                v_gear_org_id := NULL;
                v_gear_is_nbr := NULL;
                v_status := 'N';
              END IF;

            ELSE
              /* if gear id is missing, write error to error log */
              v_gear_org_id := NULL;
              v_gear_is_nbr := NULL;
            END IF;

          ELSIF v_item_name = 'Gear Configuration ID' THEN
            v_item := UPPER(v_item); -- dcl id's need to always be uppers
            IF v_item IS NOT NULL AND v_trip_qc != 1 THEN
              /* kms - assign gear to temporary key that will limit the valid gear configurations
                 to those that are of this gear type */
              v_temp_key.is_number := v_gear_is_nbr;
              v_temp_key.org_id := v_gear_org_id;

              /* get the gear configuration is number - check translations and STORET tables */
              v_storet_key := validate_storet_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_temp_key,
                                                   p_org_id,v_line_cnt,v_activity_seq);

              IF v_storet_key.is_number IS NOT NULL THEN
                v_gearcfg_is_nbr := v_storet_key.is_number;
                v_gearcfg_org_id := v_storet_key.org_id;
              ELSE
                v_gearcfg_is_nbr := NULL;
                v_gearcfg_org_id := NULL;
                v_status := 'N';
              END IF;

            ELSE
              v_gearcfg_is_nbr := NULL;
              v_gearcfg_org_id := NULL;
            END IF;

            /* kms - null out temporary key */
            v_temp_key.is_number := NULL;
            v_temp_key.org_id := NULL;

          ELSIF v_item_name = 'Sample Preservation Transport and Storage ID' THEN
            IF v_item IS NOT NULL AND v_trip_qc != 1 THEN

              /* get the sample preservation proc is number - check translations and STORET tables */
              v_storet_key := validate_storet_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_dummy_key,
                                                   p_org_id,v_line_cnt,v_activity_seq);

              IF v_storet_key.is_number IS NOT NULL THEN
                v_sample_pres_is_nbr := v_storet_key.is_number;
                v_sample_pres_org_id := v_storet_key.org_id;
              ELSE
                v_sample_pres_is_nbr := NULL;
                v_sample_pres_org_id := NULL;
                v_status := 'N';
              END IF;

            ELSE
              v_sample_pres_is_nbr := NULL;
              v_sample_pres_org_id := NULL;
            END IF;

          ELSIF v_item_name = 'Field Set ID' AND v_new_activity = 1 THEN
            v_item := UPPER(v_item); -- dcl id's need to always be uppers
            IF v_trip_qc != 1 THEN
              v_field_set_nbr := v_item;
            ELSE
              v_field_set_nbr := NULL;
            END IF;

          ELSIF v_item_name = 'Field Set Name' AND v_new_activity = 1 THEN
            IF v_trip_qc != 1 THEN
              v_field_set_name := v_item;
            ELSE
              v_field_set_name := NULL;
            END IF;

          ELSIF v_item_name = 'Characteristic' AND v_analysis_seq IS NULL THEN
            IF v_item IS NOT NULL THEN

              /* get the characteristic is number - check translations and STORET tables */
              v_storet_key := validate_storet_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_dummy_key,
                                                   p_org_id,v_line_cnt,v_activity_seq);

              IF v_storet_key.is_number IS NOT NULL THEN
                v_char_is_nbr := v_storet_key.is_number;
                v_char_org_id := v_storet_key.org_id;

                /** Assign Result type value to global Package variable **/
                v_char_result_type := get_char_result_type(v_char_is_nbr);
              ELSE
                v_char_is_nbr := NULL;
                v_char_org_id := NULL;
                v_status := 'N';
              END IF;

            ELSE
              v_char_is_nbr := NULL;
              v_char_org_id := NULL;
            END IF;

          ELSIF v_item_name = 'Characteristic Units' AND v_analysis_seq IS NULL THEN
            IF v_item IS NOT NULL THEN

              /* get the unit is number - check translations and STORET tables */
              v_storet_key := validate_storet_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_dummy_key,
                                                   p_org_id,v_line_cnt,v_activity_seq);

              IF v_storet_key.is_number IS NULL THEN
                /* if unit of measure could not be found then set org to NULL and update the status of the activity */
                v_uom_is_nbr := NULL;
                v_uom_org_id := NULL;
                v_status := 'N';
              ELSE
                v_uom_is_nbr := v_storet_key.is_number;
                v_uom_org_id := v_storet_key.org_id;
                v_success := check_uom_type(v_uom_is_nbr,v_analysis_seq,v_activity_seq);

                IF v_success = 2 THEN
                  v_status := 'N';
                  Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||':  unit of measure type does not match the unit of measure type for the indicated characteristic '||v_char_is_nbr||', line: '||v_line_cnt,NULL,v_activity_seq,NULL);
                END IF;

              END IF;
            ELSE
              v_uom_is_nbr := NULL;
              v_uom_org_id := NULL;
            END IF;

          ELSIF v_item_name = 'Sample Fraction' AND v_analysis_seq IS NULL THEN
            /* kms - validate fraction only if loaded from txt, not the analysis code. */
            IF v_item IS NOT NULL THEN

              v_fraction := validate_fraction(v_item,p_imp_cfg_seq,v_col_seq,v_char_is_nbr,v_activity_seq);

              IF v_fraction IS NULL THEN
                v_status := 'N';
              END IF;

            ELSE
              v_fraction := NULL;
            END IF;

          ELSIF v_item_name = 'Value Type' AND v_analysis_seq IS NULL THEN
            IF v_item IS NOT NULL THEN

              v_value_type_seq := validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq);

              IF v_value_type_seq IS NULL THEN
                v_status := 'N';
              END IF;

            ELSE
              v_item := 'Actual';
              Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Warning: Null value for value type has resulted in the assignment of a default value of "Actual".',NULL,v_value_type_seq,NULL);
              v_value_type_seq := validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq);
            END IF;

          ELSIF v_item_name = 'Statistic Type' AND v_analysis_seq IS NULL THEN
            IF v_item IS NOT NULL THEN

              v_stat_type_seq := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq);

              IF v_stat_type_seq IS NULL THEN
                v_status := 'N';
              END IF;

            ELSE
              v_stat_type_seq := NULL;
            END IF;

          ELSIF v_item_name = 'Field/Lab Procedure' AND v_analysis_seq IS NULL THEN
            IF v_item IS NOT NULL THEN

              /* get the analysis procedure is number - check translations and STORET tables */
              v_storet_key := validate_storet_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_dummy_key,
                                                   p_org_id,v_line_cnt,v_activity_seq);

              IF v_storet_key.is_number IS NOT NULL AND v_char_is_nbr IS NOT NULL THEN

                /* make sure procedure associated with characteristic and org */
                v_success := check_proc_char_assoc(v_storet_key,v_char_is_nbr,v_char_org_id,
                                                   v_org_is_nbr,v_activity_seq);

                /* if not associated, write error message, v_storet_key values already set to null by check_proc_char_assoc */
                IF v_success = 2 THEN
                  Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Field/Lab procedure, '||v_item||' has NOT been associated WITH the characteristic FOR this result.',NULL,v_activity_seq,NULL);
                END IF;

                v_anlpr_is_nbr := v_storet_key.is_number;
                v_anlpr_org_id := v_storet_key.org_id;
              ELSIF v_storet_key.is_number IS NOT NULL THEN

                v_anlpr_is_nbr := v_storet_key.is_number;
                v_anlpr_org_id := v_storet_key.org_id;

              ELSE
                v_anlpr_is_nbr := NULL;
                v_anlpr_org_id := NULL;
                v_status := 'N';
              END IF;

            ELSE
              v_anlpr_is_nbr := NULL;
              v_anlpr_org_id := NULL;
            END IF;

          ELSIF v_item_name = 'Lab Sample Prep Procedure' AND v_analysis_seq IS NULL THEN
            IF v_item IS NOT NULL THEN

              /* get the lab preparation procedure is number - check translations and STORET tables */
              v_storet_key := validate_storet_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_dummy_key,
                                                   p_org_id,v_line_cnt,v_activity_seq);

              IF v_storet_key.is_number IS NOT NULL THEN
                v_labprep_is_nbr := v_storet_key.is_number;
                v_labprep_org_id := v_storet_key.org_id;
              ELSE
                v_labprep_is_nbr := NULL;
                v_labprep_org_id := NULL;
                v_status := 'N';
              END IF;

            ELSE
              v_labprep_is_nbr := NULL;
              v_labprep_org_id := NULL;
            END IF;

          ELSIF v_item_name = 'Duration Basis' AND v_analysis_seq IS NULL THEN
            IF v_item IS NOT NULL THEN

              v_duration_seq := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq);

              IF v_duration_seq IS NULL THEN
                v_status := 'N';
              END IF;

            ELSE
              v_duration_seq := NULL;
            END IF;

          ELSIF v_item_name = 'Temperature Basis' AND v_analysis_seq IS NULL THEN
            IF v_item IS NOT NULL THEN

              v_temp_basis_seq := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq);

              IF v_temp_basis_seq IS NULL THEN
                v_status := 'N';
              END IF;

            ELSE
              v_temp_basis_seq := NULL;
            END IF;

       ELSIF v_item_name = 'Result Value' THEN

            IF v_item IS NOT NULL THEN
            /*SF - 10/12/01 If it's a PV type char we'll check the translation later. If we do it now, and it's empty
            we get a null value in v_result. That would be a bad thing. */
              IF v_char_result_type IN ('PV  ')  THEN
                v_result := v_item;
              ELSE
                v_result := check_translations(v_item,p_imp_cfg_seq,v_col_seq,v_dummy);
              END IF;
            ELSE
              v_status := 'N';
              Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Result Value is required and has not been provided.',NULL,v_activity_seq,NULL);
            END IF;

          ELSIF v_item_name = 'Lab ID' THEN
            v_item := UPPER(v_item); -- dcl id's need TO always be uppers
            IF v_item IS NOT NULL THEN

              /* get the lab is number - check translations and STORET tables */
              v_storet_key := validate_storet_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_dummy_key,
                                                   p_org_id,v_line_cnt,v_activity_seq);

              IF v_storet_key.is_number IS NOT NULL THEN
                v_tsrlab_is_nbr := v_storet_key.is_number;
                v_tsrlab_org_id := v_storet_key.org_id;
              ELSE
                v_tsrlab_is_nbr := NULL;
                v_tsrlab_org_id := NULL;
                v_status := 'N';
              END IF;

            ELSE
              v_tsrlab_is_nbr := NULL;
              v_tsrlab_org_id := NULL;
            END IF;

          ELSIF v_item_name = 'Analysis Date' THEN
            IF v_item IS NOT NULL THEN
              v_analysis_date := get_date(v_item_name,v_item,v_col_seq,p_imp_cfg_seq,v_activity_seq);

              IF v_analysis_date IS NULL THEN
                Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Warning: Analysis Date format not recognized.',NULL,v_activity_seq,NULL);
              END IF;

            ELSE
              v_analysis_date := NULL;
            END IF;

          ELSIF v_item_name = 'Analysis Time' THEN
            IF v_analysis_date IS NOT NULL THEN
              IF v_item IS NOT NULL THEN
                v_analysis_time := get_time(v_item_name,v_item,v_col_seq,p_imp_cfg_seq,v_activity_seq);

                IF v_analysis_time IS NULL THEN
                  Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Warning: Analysis Time format not recognized.',NULL,v_activity_seq,NULL);
                END IF;

              ELSE
                v_analysis_time := NULL;
              END IF;
            ELSE
              v_analysis_time := NULL;
            END IF;

          ELSIF v_item_name = 'Quantification Low' THEN
            IF v_item IS NOT NULL THEN
              v_lowq_limit := v_item;
            ELSE
              v_lowq_limit := NULL;
            END IF;

          ELSIF v_item_name = 'Quantification High' THEN
            IF v_item IS NOT NULL THEN
              v_upq_limit := v_item;
            ELSE
              v_upq_limit := NULL;
            END IF;

          ELSIF v_item_name = 'Detection Limit' THEN
            IF v_item IS NOT NULL THEN
              v_detect_limit := v_item;
            ELSE
              v_detect_limit := NULL;
            END IF;

          ELSIF v_item_name = 'Detection Limit Unit' THEN
            IF v_item IS NOT NULL AND (v_detect_limit IS NOT NULL OR v_upq_limit IS NOT NULL OR v_lowq_limit IS NOT NULL) THEN
              /* get the detection unit is number - check translations and STORET tables */
             IF NVL(v_char_result_type,'x') <> 'TEXT' THEN -- IF added by DCL to avoid unit of measure error if result is a text type
                v_storet_key := validate_storet_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_dummy_key,
                                                   p_org_id,v_line_cnt,v_activity_seq);
             END IF;

              IF v_storet_key.is_number IS NULL THEN
                /* if unit of measure could not be found then set org to NULL and update the status of the activity */
                v_det_uom_is_nbr := NULL;
                v_det_uom_org_id := NULL;
                v_status := 'N';

              ELSE
                v_det_uom_is_nbr := v_storet_key.is_number;
                v_det_uom_org_id := v_storet_key.org_id;
                /* kms - need to make sure unit types match */
                v_success := check_uom_type(v_det_uom_is_nbr,v_analysis_seq,v_activity_seq);

                IF v_success = 2 THEN
                  v_status := 'N';
                  Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Detection limit unit of measure type does not match the unit of measure type for the indicated characteristic.',NULL,v_activity_seq,NULL);
                END IF;
              END IF;
            ELSIF v_item IS NOT NULL THEN
              IF v_detect_limit IS NULL AND v_upq_limit IS NULL AND v_lowq_limit IS NULL THEN
                sim2_import_pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||' Warning: Detection limit unit of measure has been provided and is not used. It has been removed.',null,v_activity_seq,null);
                v_det_uom_is_nbr := NULL;
                v_det_uom_org_id := NULL;
              END IF;
            ELSE
              v_det_uom_is_nbr := NULL;
              v_det_uom_org_id := NULL;
            END IF;
            
          ELSIF v_item_name = 'Detection Limit Comment' THEN
            v_det_comment := SUBSTR(v_item,1,254);

          ELSIF v_item_name = 'Precision' THEN
            v_precision := v_item;

          ELSIF v_item_name = 'Confidence Level' THEN
            IF v_item IS NOT NULL THEN
              v_conflvl_seq := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq);

              IF v_precision IS NULL THEN
                v_status := 'N';
                Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Confidence level has been provided without a value for precision, activity will not be exported.',NULL,v_activity_seq,NULL);
              END IF;

            ELSE
              v_conflvl_seq := NULL;
            END IF;

          ELSIF v_item_name = 'Result Comment' THEN

             IF NVL(v_char_result_type,'x') = 'TEXT' AND v_item IS NOT NULL THEN
               /** dcl 09/13/01 if a result comment exists for a text type result **/
                Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Result Comments are not used with a Text-type result - comment ignored',NULL,v_activity_seq,NULL);
                v_result_cmt := NULL;
             ELSE
                v_result_cmt := v_item;
             END IF;

          ELSIF v_item_name = 'Result Status' THEN
            v_item := UPPER(v_item); -- dcl status always uppers
            IF v_item IS NOT NULL THEN
              v_result_status_seq := validate_alval_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_activity_seq);
            ELSE
              v_result_status_seq := NULL;
              Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning: Result Status has been left blank and will be set to ''Final'' when exported.',NULL,v_activity_seq,NULL);
            END IF;

          ELSIF v_item_name = 'Batch ID' THEN
            v_item := UPPER(v_item); -- dcl id's need to always be uppers
            v_batch_id := v_item;

          ELSIF v_item_name = 'Laboratory Certified' THEN
            v_lab_cert := v_item;

          ELSIF v_item_name = 'Project ID' AND v_new_activity = 1 THEN
            IF v_item IS NOT NULL THEN
              v_item := UPPER(v_item); -- dcl ID's always uppers
              v_no_projects := 0;

              /* can have multiple projects separated by '\' - get number of projects to expect */
              v_no_projects := LENGTH(v_item) - LENGTH(TRANSLATE(v_item,'a\','a')) + 1;

              /* separate item into projects using parse procedure if more than 1 project */
              IF v_no_projects > 1 THEN
                BEGIN
                  Sim2_Import_Pkg.parse_line_table(v_item,v_project_table,v_no_projects,'\',v_bad_parse,v_import_log_seq);
                EXCEPTION
                  WHEN NO_DATA_FOUND THEN
                    NULL;
                END;
              ELSE
                v_project_table(1) := v_item;
              END IF;

              /* loop through projects table and add each project in column to the activity */
              FOR v_number IN 1..v_no_projects LOOP
                /* Check that Project ID is not longer than 8 characters -- sfp 061101 */
                /* kms (6/21/01) - changed to check length of item in project table rather than
                   of v_item. */
                IF LENGTH(v_project_table(v_number)) > 8 THEN
                  v_status := 'N';
                  Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Project ID ('||v_project_table(v_number)||') must not contain more than 8 characters.',NULL,v_activity_seq,NULL);
                END IF;

                /* get the project is number - check translations and STORET tables */
                v_storet_key := validate_storet_item(v_item_name,v_project_table(v_number),p_imp_cfg_seq,v_col_seq,
                                                     v_dummy_key,p_org_id,v_line_cnt,v_activity_seq);

                IF v_storet_key.is_number IS NOT NULL THEN
                  v_proj_is_table(v_number) := v_storet_key.is_number;
                  v_proj_org_table(v_number) := v_storet_key.org_id;
                ELSE
                  /* create project if project id not found*/
                  /* sfp 061101 - creating projects during import is no longer allowed. */
                  /* v_proj_org_table(v_number) := RPAD(p_org_id,8); */
                  /* v_proj_is_table(v_number) := add_project(v_project_table(v_number),p_org_id,v_activity_seq); */
                  v_status := 'N';
                END IF;

              END LOOP;

            ELSE
              /* if no project id, write error to error log and set status to 'N' */
              v_proj_is_nbr := NULL;
              v_proj_org_id := NULL;
              v_status := 'N';
              Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Project ID is required and has not been provided.',NULL,v_activity_seq,NULL);
            END IF;

          ELSIF v_item_name = 'Characteristic Group ID' THEN
            v_item := UPPER(v_item); -- dcl id's need to always be uppers
            IF v_item IS NOT NULL THEN

              /* get the char group is number - check translations and STORET tables */
              v_storet_key := validate_storet_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_dummy_key,
                                                   p_org_id,v_line_cnt,v_activity_seq);

              IF v_storet_key.is_number IS NOT NULL THEN
                /* kms - make sure group is valid for this activity.  Check type and medium. */
                v_success := check_group_type_med(v_storet_key,v_type_seq,v_medium_seq,v_activity_seq);

                IF v_success != 1 THEN
                  /* kms - group is not valid for this activity or an error occured when checking */
                  v_tsrchgrp_is_nbr := NULL;
                  v_tsrchgrp_org_id := NULL;
                  v_status := 'N';
                ELSE
                  /* kms - group is valid for this activity */
                  v_tsrchgrp_is_nbr := v_storet_key.is_number;
                  v_tsrchgrp_org_id := v_storet_key.org_id;
                END IF;
              ELSE
                v_tsrchgrp_is_nbr := NULL;
                v_tsrchgrp_org_id := NULL;
                v_status := 'N';
              END IF;

            ELSE
              v_tsrchgrp_is_nbr := NULL;
              v_tsrchgrp_org_id := NULL;
            END IF;

          ELSIF v_item_name = 'Characteristic Row ID' THEN
            IF v_item IS NOT NULL AND v_tsrchgrp_is_nbr IS NOT NULL THEN
              v_row_id := v_item;

              /* kms - set temp key to group pk values in order to send to validation routine */
              v_temp_key.is_number := v_tsrchgrp_is_nbr;
              v_temp_key.org_id := v_tsrchgrp_org_id;

              /* get the char row is number - check translations and STORET tables */
              v_storet_key := validate_storet_item(v_item_name,v_item,p_imp_cfg_seq,v_col_seq,v_temp_key,
                                                   p_org_id,v_line_cnt,v_activity_seq);

              IF v_storet_key.is_number IS NOT NULL THEN
                v_tsrchdef_is_nbr := v_storet_key.is_number;
                v_tsrchdef_org_id := v_storet_key.org_id;

                /* populate result char fields based on what's in the default table */
                v_dummy_seq := NULL;
                v_success := get_char_defaults(v_dummy_seq, v_storet_key, p_imp_cfg_seq, v_col_seq, v_activity_seq);

                /* assign defaults to result variables */
                v_char_is_nbr := v_acode_defs_rec.char_pk.is_number;
                v_char_org_id := v_acode_defs_rec.char_pk.org_id;
                v_uom_is_nbr := v_acode_defs_rec.uom_pk.is_number;
                v_uom_org_id := v_acode_defs_rec.uom_pk.org_id;
                v_anlpr_is_nbr := v_acode_defs_rec.anlpr_pk.is_number;
                v_anlpr_org_id := v_acode_defs_rec.anlpr_pk.org_id;
                v_labprep_is_nbr := v_acode_defs_rec.labprep_pk.is_number;
                v_labprep_org_id := v_acode_defs_rec.labprep_pk.org_id;
                v_fraction := v_acode_defs_rec.fraction;
                v_value_type_seq := v_acode_defs_rec.value_type;
                v_stat_type_seq := v_acode_defs_rec.stat_type;
                v_duration_seq := v_acode_defs_rec.dur_basis;
                v_temp_basis_seq := v_acode_defs_rec.temp_basis;

              ELSE
                v_tsrchdef_is_nbr := NULL;
                v_tsrchdef_org_id := NULL;
                v_status := 'N';
              END IF;
            ELSIF v_item IS NOT NULL AND v_tsrchgrp_is_nbr IS NULL THEN
              /* kms - importing characteristic row, but there's no characteristic group
                 to validate it with.  Invalidate the activity and write error message. */
              v_tsrchdef_is_nbr := NULL;
              v_tsrchdef_org_id := NULL;
              v_status := 'N';
              Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Characteristic Row ID has been included but cannot be validated due to missing or invalid Characteristic Group ID.',NULL,v_activity_seq,NULL);
            ELSIF v_item IS NULL AND v_tsrchgrp_is_nbr IS NOT NULL THEN
              /* kms - characteristic group has been imported, but no row.  Invalidate activity
                 and write error message. */
              v_tsrchdef_is_nbr := NULL;
              v_tsrchdef_org_id := NULL;
              v_status := 'N';
              Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Characteristic Row ID is required when Characteristic Group ID has been given.',NULL,v_activity_seq,NULL);
            ELSE
              v_tsrchdef_is_nbr := NULL;
              v_tsrchdef_org_id := NULL;
           END IF;

          ELSIF v_item_name = 'AnalCode' THEN
            NULL;
          /**** DTW Not Supporting Analysis Codes in this version
            IF v_item IS NOT NULL THEN
              v_analysis_code := SUBSTR(v_item,1,15);

              -- check the analysis code table and get the sequence
              v_analysis_seq := get_analysis_code_seq(v_activity_seq,v_analysis_code,v_status,
                                                      v_tsrlab_is_nbr, v_tsrlab_org_id, v_line_cnt,v_activity_seq);

              -- make sure analysis code is complete and valid
              v_success := check_acode_comp(v_analysis_seq, v_activity_seq);
              IF v_success = 2 THEN
                -- if analysis code is not complete, prevent activity from being exported
                v_status := 'N';
              END IF;

              -- populate result char fields based on what's in the analysis codes/default tables
              v_dummy_key.is_number := NULL;
              v_dummy_key.org_id := NULL;
              v_success := get_char_defaults(v_analysis_seq,v_dummy_key,p_imp_cfg_seq, v_col_seq, v_activity_seq);

              -- assign defaults to result variables
              v_char_is_nbr := v_acode_defs_rec.char_pk.is_number;
              v_char_org_id := v_acode_defs_rec.char_pk.org_id;
              v_tsrlab_is_nbr := v_acode_defs_rec.lab_pk.is_number;
              v_tsrlab_org_id := v_acode_defs_rec.lab_pk.org_id;
              v_uom_is_nbr := v_acode_defs_rec.uom_pk.is_number;
              v_uom_org_id := v_acode_defs_rec.uom_pk.org_id;
              v_anlpr_is_nbr := v_acode_defs_rec.anlpr_pk.is_number;
              v_anlpr_org_id := v_acode_defs_rec.anlpr_pk.org_id;
              v_labprep_is_nbr := v_acode_defs_rec.labprep_pk.is_number;
              v_labprep_org_id := v_acode_defs_rec.labprep_pk.org_id;
              v_fraction := v_acode_defs_rec.fraction;
              v_value_type_seq := v_acode_defs_rec.value_type;
              v_stat_type_seq := v_acode_defs_rec.stat_type;
              v_duration_seq := v_acode_defs_rec.dur_basis;
              v_temp_basis_seq := v_acode_defs_rec.temp_basis;

            ELSE
              -- if no analysis code, write error to error log and set status to 'N'
              v_analysis_seq := NULL;
            END IF;

            DTW ****/
          END IF;


        END IF;

        END LOOP; /* end of column loop */

           /* bka (8-20-01) check to see if value type was included in row,
             if not, then add it with a default value of 'Actual' */
          IF v_value_type_seq IS NULL THEN
            v_item_name := 'Value Type';
            v_item      := 'Actual';
            v_col_seq   := get_col_seq('Value Type', p_imp_cfg_seq, NULL);
            v_value_type_seq := validate_alval_item(v_item_name, v_item, p_imp_cfg_seq, v_col_seq, v_activity_seq);
            Sim2_Import_Pkg.write_error(v_import_log_seq,'Line '||v_line_cnt||': Warning: Value type is required for all results, and has resulted in the assignment of a default value type of "Actual".',NULL,v_value_type_seq,NULL);
          END IF;

        /* see if activity already exists in STORET */
        v_storet_key := check_storet_act_id(v_activity_id, v_replicate_nbr, v_trip_is_nbr, v_tsmstatn_is_nbr, p_org_id, v_activity_seq);

        IF v_storet_key.is_number IS NOT NULL THEN
          /* activity already exists in STORET, create in sim but do not allow it to be exported */
          v_act_is_nbr := v_storet_key.is_number;
          v_act_org_id := v_storet_key.org_id;
          v_status := 'N';
          Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||v_activity_id||' already exists in STORET, this activity will not be exported.',NULL,v_activity_seq,NULL);
        ELSE
          v_act_is_nbr := NULL;
          v_act_org_id := NULL;
        END IF;

        /* kms (6/28/01) - make sure field set id is included if field set name is there and
           vice versa. */
        IF v_field_set_nbr IS NOT NULL AND v_field_set_name IS NULL THEN
          v_status := 'N';
          Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Field SET Name must be included when field set id is present.',NULL,v_activity_seq,NULL);
        ELSIF v_field_set_nbr IS NULL AND v_field_set_name IS NOT NULL THEN
          v_status := 'N';
          Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Field set ID must be included when field set name is present.',NULL,v_activity_seq,NULL);
        END IF;

        /* see if result already exists on activity (in SIM)  */
        IF NOT check_duplicate_result(v_char_is_nbr,v_activity_seq,v_fraction,v_detect_limit,v_uom_is_nbr,v_anlpr_is_nbr,v_labprep_is_nbr,v_duration_seq, v_temp_basis_seq,v_precision,v_conflvl_seq,v_stat_type_seq) THEN
          v_status := 'N';
        END IF;

        /* make sure field gear types in the gear and collection procedure are the same */
        IF v_proc_is_nbr IS NOT NULL THEN
          --BEGIN
            /* see if gear is required, compare gear types and return 0 if they match */
            v_gear_type_ok := check_gear_type(v_proc_is_nbr, v_gear_is_nbr, p_org_id, v_activity_seq);

            IF v_gear_type_ok = 1 THEN
              /* gear types do not match, write error to log and change status to 'N' */
              v_status := 'N';
              Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Gear types for collection procedure and gear do not match.',NULL,v_activity_seq,NULL);
            ELSIF v_gear_type_ok = 2 THEN
              /* sample collection procedure has a gear type but gear has not been provided, invalidate activity
                 and write error message */
              v_status := 'N';
              Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Gear is required when the collection procedure has a gear type.',NULL,v_activity_seq,NULL);
            END IF;
--SF - 10/5/01 Took this out to make the package smaller.
          --EXCEPTION
            --WHEN OTHERS THEN
              --Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - checking gear types.',NULL,v_activity_seq,NULL);
          --END;

        ELSIF v_proc_is_nbr IS NULL AND v_new_activity = 1 AND v_type = 'Sample' THEN
          /* write error if sample collection procedure has not been included for a sample */
          v_status := 'N';
          Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Sample Collection procedure is required and has not been included or is invalid.',NULL,v_activity_seq,NULL);
        END IF;

        /* make sure detection limits have a unit of measure */
        IF v_detect_limit IS NOT NULL AND v_det_uom_is_nbr IS NULL THEN
          Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Detection limit unit of measure has not been provided, and is required when detection limit is given.',NULL,v_activity_seq,NULL);
          v_status := 'N';
        END IF;
        IF v_lowq_limit IS NOT NULL AND v_det_uom_is_nbr IS NULL THEN
          Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Detection limit unit of measure has not been provided, and is required when lower quantification limit is given.',NULL,v_activity_seq,NULL);
          v_status := 'N';
        END IF;
        IF v_upq_limit IS NOT NULL AND v_det_uom_is_nbr IS NULL THEN
          Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Detection limit unit of measure has not been provided, and is required when upper quantification limit is given.',NULL,v_activity_seq,NULL);
          v_status := 'N';
        END IF;
        --sef - this is wrong. It's not the result value that matters it's the q-limits. Fixed in previous two IF blocks
        /* bka - added to make sure that detection UOM is present when detection limit results are imported */
        --IF v_result IN ('*Present >QL','*Present <QL') AND v_det_uom_is_nbr IS NULL THEN
          --Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Detection limit unit of measure has not been provided, and is required when a result value of *Present >QL or *Present <QL is given.',NULL,v_activity_seq,NULL);
          --v_status := 'N';
        --END IF;

        /* check the characteristic type, if physical or permitted value type, perform
           further checks on result. If the char. type is text, copy result value into
           the result comment part of the code. */

        v_result_save := v_result;
        v_result := update_result(v_result,v_char_is_nbr,v_units_ok,v_result_ok,p_imp_cfg_seq,v_activity_seq);
        IF v_char_result_type IN ('TEXT') THEN
           v_result_cmt := v_result_save;
           v_result_save := NULL; -- reset after each use.
        END IF;
        IF v_char_result_type IN ('PHYS') THEN
          IF v_result IN ('*Present >QL')  THEN
            Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Physical Characteristic cannot have a non-numberical result value of *Present >QL.',NULL,v_activity_seq,NULL);
            v_status := 'N';
          ELSIF v_result IN ('*Present <QL')  THEN
            Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Physical Characteristic cannot have a non-numberical result value of *Present >QL.',NULL,v_activity_seq,NULL);
            v_status := 'N';
          ELSIF v_result IN ('*Non-detect')  THEN
            Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Physical Characteristic cannot have a non-numberical result value of *Non-detect.',NULL,v_activity_seq,NULL);
            v_status := 'N';
          ELSIF v_result IN ('*Present')  THEN
            Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Physical Characteristic cannot have a non-numberical result value of *Present.',NULL,v_activity_seq,NULL);
            v_status := 'N';
          END IF;
        END IF;
        IF NOT v_result_ok THEN
          v_status := 'N';
        ELSIF v_result = '0' THEN
          v_result_value_msr := 0;
        ELSIF v_result IS NULL THEN
          v_status := 'N';
        END IF;

        IF v_char_is_nbr IS NULL AND v_analysis_seq IS NULL THEN
          v_status := 'N';
          Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Either a characteristic or characteristic group must be provided.',NULL,v_activity_seq,NULL); -- Changed "analysis code" to "characteristic group" since analysis codes are no longer supported

        ELSIF v_char_is_nbr IS NOT NULL THEN
          v_success := check_chardet_comp(v_char_is_nbr, v_char_org_id, v_anlpr_is_nbr, v_uom_is_nbr,
                                          v_uom_org_id, v_fraction, v_value_type_seq, v_units_ok,
                                          v_activity_seq);

          IF v_success = 2 THEN
            v_status := 'N';
          ELSIF v_success = 3 THEN
            v_uom_is_nbr := NULL;
            v_uom_org_id := NULL;
          END IF;

        END IF;

        /* create activity */
        IF v_new_activity = 1 THEN
          BEGIN
            IF v_status = 'N' THEN
              v_approved_on := NULL;
              v_approved_by := NULL;
            ELSE
              v_approved_on := SYSDATE;
              v_approved_by := USER;
            END IF;
          EXCEPTION
            WHEN OTHERS THEN
              Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - in approved part',NULL,v_activity_seq,NULL);
          END;

          BEGIN
            /* assign values to record in order to send all of these variables to function easily */
            v_fa_rec.FA_SEQ := v_activity_seq;
            v_fa_rec.FA_TYPE_SEQ := v_type_seq;
            v_fa_rec.FA_CATEGORY_SEQ :=  v_cat_seq;
            v_fa_rec.FA_ACTIVITY_ID := v_activity_id;
            v_fa_rec.FA_START_DATE := v_start_date;
            v_fa_rec.FA_START_TIME := v_start_time;
            v_fa_rec.FA_STOP_DATE := v_end_date;
            v_fa_rec.FA_STOP_TIME := v_end_time;
            v_fa_rec.FA_DEPTH := v_depth;
            v_fa_rec.FA_DEPTH_REF_POINT_SEQ := v_depth_ref;
            v_fa_rec.FA_ACTIVITY_COMMENTS := v_comment;
            v_fa_rec.FA_TSMSTATN_IS_NUMBER := v_tsmstatn_is_nbr;
            v_fa_rec.FA_TSMSTATN_ORG_ID := v_tsmstatn_org_id;
            v_fa_rec.FA_TSRTRIP_IS_NUMBER := v_trip_is_nbr;
            v_fa_rec.FA_TSRTRIP_ORG_ID := v_trip_org_id;
            v_fa_rec.FA_TSRSTVST_IS_NUMBER := v_tsrstvst_is_nbr;
            v_fa_rec.FA_TSRSTVST_ORG_ID := v_tsrstvst_org_id;
            v_fa_rec.FA_TSRFLDPR_IS_NUMBER := v_proc_is_nbr;
            v_fa_rec.FA_TSRFLDPR_ORG_ID := v_proc_org_id;
            v_fa_rec.FA_TSRFLDGR_IS_NUMBER := v_gear_is_nbr;
            v_fa_rec.FA_TSRFLDGR_ORG_ID := v_gear_org_id;
            v_fa_rec.FA_IMPORT_SEQ := v_import_log_seq;
            v_fa_rec.FA_EXPORT_SEQ := NULL;
            v_fa_rec.FA_STATUS := v_status;
            v_fa_rec.FA_TSMORGAN_IS_NUMBER := v_org_is_nbr;
            v_fa_rec.FA_TSMORGAN_ORG_ID := p_org_id;
            v_fa_rec.FA_APPROVED_BY := v_approved_by;
            v_fa_rec.FA_APPROVED_ON := v_approved_on;
            v_fa_rec.FA_STVST_COMMENTS := v_stvst_cmt;
            v_fa_rec.FA_DU_SEQ := v_depth_unit_seq;
            v_fa_rec.FA_MED_SEQ := v_medium_seq;
            v_fa_rec.FA_TSRLAB_IS_NUMBER := v_tsrlab_is_nbr;
            v_fa_rec.FA_TSRLAB_ORG_ID := v_tsrlab_org_id;
            v_fa_rec.FA_IMPORT_LINE_NO := v_line_cnt;
            v_fa_rec.FA_TSRFDACT_IS_NUMBER := v_act_is_nbr;
            v_fa_rec.FA_TSRFDACT_ORG_ID := v_act_org_id;
            v_fa_rec.FA_REPLICATE_NBR := v_replicate_nbr;
            v_fa_rec.FA_FDSET_ID := v_field_set_nbr;
            v_fa_rec.FA_FDSET_NAME := v_field_set_name;
            v_fa_rec.FA_UPDEPTH := v_updepth;
            v_fa_rec.FA_LOWDEPTH := v_lowdepth;
            v_fa_rec.FA_THERMO_SEQ := v_thermo_seq;
            v_fa_rec.FA_HALO_SEQ := v_halo_seq;
            v_fa_rec.FA_PYCNO_SEQ := v_pycno_seq;
            v_fa_rec.FA_REL_DEPTH_SEQ := v_reldepth_seq;
            v_fa_rec.FA_DEPTHZONE_SEQ := v_depthzone_seq;
            v_fa_rec.FA_DEPTHRU_SEQ := v_drange_seq;
            v_fa_rec.FA_TIME_ZONE_SEQ := v_time_zone;
            v_fa_rec.FA_TSRGRCFG_IS_NUMBER := v_gearcfg_is_nbr;
            v_fa_rec.FA_TSRGRCFG_ORG_ID := v_gearcfg_org_id;
            v_fa_rec.FA_TSRSDP_IS_NUMBER := v_sample_pres_is_nbr;
            v_fa_rec.FA_TSRSDP_ORG_ID := v_sample_pres_org_id;
            v_fa_rec.FA_STP_SEQ := v_trip_seq;
            v_fa_rec.FA_SSV_SEQ := v_visit_seq;

            /* insert field activity */
            v_success := create_field_activity(v_fa_rec);

            IF v_success != 1 THEN
              UPDATE SIM_FIELD_ACTIVITIES
                SET fa_status = 'N'
              WHERE fa_seq = v_activity_seq;

              Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Failed to create activity in sim_field_activities, activity will not be exported.',NULL,v_activity_seq,NULL);

            END IF;

          EXCEPTION
            WHEN OTHERS THEN
              v_import_desc := 'Line '||v_line_cnt||': '||SQLERRM||' - creating field activity.';
              Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, v_import_desc,NULL,v_activity_seq,NULL);
          END;

          /* add personnel and projects to activity */
          BEGIN
            IF v_no_projects IS NOT NULL THEN
              FOR v_number IN 1..v_no_projects LOOP

                IF v_proj_is_table(v_number) IS NOT NULL THEN
                  v_proj_is_nbr := v_proj_is_table(v_number);
                  v_proj_org_id := v_proj_org_table(v_number);

                  v_success := add_project_to_activity(v_activity_seq, v_proj_is_nbr, v_proj_org_id);
                END IF;
              END LOOP;
            ELSE
              v_status := 'N';
              UPDATE SIM_FIELD_ACTIVITIES
                SET FA_STATUS = 'N'
               WHERE FA_SEQ = v_activity_seq;

              Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Project is required and has not been provided.',NULL,v_activity_seq,NULL);
            END IF;

          EXCEPTION
            WHEN NO_DATA_FOUND THEN
              NULL;

            WHEN OTHERS THEN
              Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': '||SQLERRM||' - adding projects to activity.',NULL,v_activity_seq,NULL);
          END;

          BEGIN
            IF v_no_person > 0 THEN
              FOR v_number IN 1..v_no_person LOOP
                IF v_persn_is_table(v_number) IS NOT NULL THEN
                  v_tsmpersn_is_nbr := v_persn_is_table(v_number);
                  v_tsmpersn_org_id := v_persn_org_table(v_number);

                  v_success := add_personnel_to_activity(v_activity_seq, v_tsmpersn_is_nbr, v_tsmpersn_org_id);
                END IF;
              END LOOP;
            END IF;

          EXCEPTION
            WHEN NO_DATA_FOUND THEN
              NULL;

            WHEN OTHERS THEN
              Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,'Line '||v_line_cnt||': '||SQLERRM||' - adding personnel to activity.',NULL,v_activity_seq,NULL);
          END;

        END IF;

        /* create result */
        BEGIN

          /* write data to record and send to create result function */
          /* get result sequence */
          BEGIN
            OPEN c_get_result_seq;
            FETCH c_get_result_seq INTO v_result_rec.RS_SEQ;
            CLOSE c_get_result_seq;
          EXCEPTION
            WHEN OTHERS THEN
              Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - getting results sequence.',NULL,v_activity_seq,NULL);
          END;
          IF v_char_result_type NOT IN ('TEXT','PV  ')  THEN -- DCL added this to keep extraneous data from being imported if the result is text or pv

             v_result_rec.RS_AC_SEQ := v_analysis_seq;
             v_result_rec.RS_FA_SEQ := v_activity_seq;
             v_result_rec.RS_RESULT := v_result;
             v_result_rec.RS_COMMENT := v_result_cmt;
             v_result_cmt := NULL;
             v_result_rec.RS_DETECT_LIMIT := v_detect_limit;
             v_result_rec.RS_IMPORT_LINE_NO := v_line_cnt;
             v_result_rec.RS_VALUE_STATUS_SEQ := v_result_status_seq;
             v_result_rec.RS_LAB_BATCH_ID := v_batch_id;
             v_result_rec.RS_LAB_CERT := v_lab_cert;
             v_result_rec.RS_ANALYSIS_DATE := v_analysis_date;
             v_result_rec.RS_ANALYSIS_TIME := v_analysis_time;
             v_result_rec.RS_DET_LIMIT_UOM_NBR := v_det_uom_is_nbr;
             v_result_rec.RS_DET_LIMIT_UOM_ORG := v_det_uom_org_id;
             v_result_rec.RS_DETECT_COMMENT := v_det_comment;
             v_result_rec.RS_UPPER_QUANT_LIMIT := v_upq_limit;
             v_result_rec.RS_LOWER_QUANT_LIMIT := v_lowq_limit;
             v_result_rec.RS_TSRCHAR_IS_NUMBER := v_char_is_nbr;
             v_result_rec.RS_TSRCHAR_ORG_ID := v_char_org_id;
             v_result_rec.RS_TSRUOM_IS_NBR := v_uom_is_nbr;
             v_result_rec.RS_TSRUOM_ORG_ID := v_uom_org_id;
             v_result_rec.RS_FRACTION := v_fraction;
             v_result_rec.RS_VALUE_TYPE_SEQ := v_value_type_seq;
             v_result_rec.RS_STAT_TYPE_SEQ := v_stat_type_seq;
             v_result_rec.RS_TSRANLPR_IS_NUMBER := v_anlpr_is_nbr;
             v_result_rec.RS_TSRANLPR_ORG_ID := v_anlpr_org_id;
             v_result_rec.RS_TSRLSPP_IS_NUMBER := v_labprep_is_nbr;
             v_result_rec.RS_TSRLSPP_ORG_ID := v_labprep_org_id;
             v_result_rec.RS_DURBASIS_SEQ := v_duration_seq;
             v_result_rec.RS_TEMPBASIS_SEQ := v_temp_basis_seq;
             v_result_rec.RS_PRECISION := v_precision;
             v_result_rec.RS_CONFLEVEL_SEQ := v_conflvl_seq;
             v_result_rec.RS_TSRCHGRP_IS_NUMBER := v_tsrchgrp_is_nbr;
             v_result_rec.RS_TSRCHGRP_ORG_ID := v_tsrchgrp_org_id;
             v_result_rec.RS_TSRCHDEF_IS_NUMBER := v_tsrchdef_is_nbr;
             v_result_rec.RS_TSRCHDEF_ORG_ID := v_tsrchdef_org_id;
             v_result_rec.RS_TSRLAB_IS_NUMBER := v_tsrlab_is_nbr;
             v_result_rec.RS_TSRLAB_ORG_ID := v_tsrlab_org_id;

             /***** added by dcl to manage extra data in the event of text OR pv result type ******/
             /***** any UOMs or other data that might have been included but is not needed
             /***** is nulled out at this point 09/05/01                               ****/
          ELSE
                v_result_rec.RS_AC_SEQ := v_analysis_seq;
                v_result_rec.RS_FA_SEQ := v_activity_seq;
                v_result_rec.RS_RESULT := v_result;
                v_result_rec.RS_COMMENT := v_result_cmt;
                v_result_cmt := NULL;
                IF v_result_rec.RS_DETECT_LIMIT IS NOT NULL THEN
                   v_result_rec.RS_DETECT_LIMIT := NULL;
                   Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning - Detection Limit not needed for text type result.',NULL,v_activity_seq,NULL);
                END IF;

                v_result_rec.RS_IMPORT_LINE_NO := v_line_cnt;
                v_result_rec.RS_VALUE_STATUS_SEQ := v_result_status_seq;
                v_result_rec.RS_LAB_BATCH_ID := v_batch_id;
                IF v_result_rec.RS_LAB_CERT IS NOT NULL THEN
                   v_result_rec.RS_LAB_CERT := NULL;
                   Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning - Lab Certification not needed for text type result',NULL,v_activity_seq,NULL);
                END IF;

                IF v_result_rec.RS_ANALYSIS_DATE IS NOT NULL THEN
                   v_result_rec.RS_ANALYSIS_DATE := NULL;
                   Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning - Analysis Date not needed for text type result.',NULL,v_activity_seq,NULL);
                END IF;

                IF v_result_rec.RS_ANALYSIS_TIME IS NOT NULL THEN
                   v_result_rec.RS_ANALYSIS_TIME := NULL;
                   Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning - Analysis Time not needed for text type result',NULL,v_activity_seq,NULL);
                END IF;

                IF v_result_rec.RS_DET_LIMIT_UOM_NBR IS NOT NULL THEN
                   v_result_rec.RS_DET_LIMIT_UOM_NBR := NULL;
                   v_result_rec.RS_DET_LIMIT_UOM_ORG := NULL;
                   Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning - Detection limit UOM not needed for text type result',NULL,v_activity_seq,NULL);
                END IF;

                IF v_result_rec.RS_DETECT_COMMENT IS NOT NULL THEN
                   v_result_rec.RS_DETECT_COMMENT := NULL;
                   Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning - Detect Comment not needed for text type result',NULL,v_activity_seq,NULL);
                END IF;

                IF v_result_rec.RS_UPPER_QUANT_LIMIT IS NOT NULL THEN
                   v_result_rec.RS_UPPER_QUANT_LIMIT := NULL;
                   Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning - Upper Quantificatin Limit not needed for text type result',NULL,v_activity_seq,NULL);
                END IF;

                IF v_result_rec.RS_LOWER_QUANT_LIMIT IS NOT NULL THEN
                   v_result_rec.RS_LOWER_QUANT_LIMIT := NULL;
                   Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning - Lower Quantification Limit not needed for text type result',NULL,v_activity_seq,NULL);
                END IF;

                v_result_rec.RS_TSRCHAR_IS_NUMBER := v_char_is_nbr;
                v_result_rec.RS_TSRCHAR_IS_NUMBER := v_char_is_nbr;
                v_result_rec.RS_TSRCHAR_ORG_ID := v_char_org_id;
                IF v_result_rec.RS_TSRUOM_IS_NBR IS NOT NULL THEN
                   v_result_rec.RS_TSRUOM_IS_NBR := NULL;
                   v_result_rec.RS_TSRUOM_ORG_ID := NULL;
                   Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning - Unit of Measure not needed for text type result',NULL,v_activity_seq,NULL);
                END IF;

                IF v_result_rec.RS_FRACTION IS NOT NULL THEN
                   v_result_rec.RS_FRACTION := NULL;
                   Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning - Fraction not needed for text type result',NULL,v_activity_seq,NULL);
                END IF;

                IF v_result_rec.RS_VALUE_TYPE_SEQ IS NOT NULL THEN
                   v_result_rec.RS_VALUE_TYPE_SEQ := NULL;
                   Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning - Value Type not needed for text type result',NULL,v_activity_seq,NULL);
                END IF;

                IF v_result_rec.RS_STAT_TYPE_SEQ IS NOT NULL THEN
                   v_result_rec.RS_STAT_TYPE_SEQ := NULL;
                   Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning - Statistic Type not needed for text type result',NULL,v_activity_seq,NULL);
                END IF;

                IF v_result_rec.RS_TSRANLPR_IS_NUMBER IS NOT NULL THEN
                   v_result_rec.RS_TSRANLPR_IS_NUMBER := NULL;
                   v_result_rec.RS_TSRANLPR_ORG_ID := NULL;
                   Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning - Analysis Procedure not needed for text type result',NULL,v_activity_seq,NULL);
                END IF;

                IF v_result_rec.RS_TSRLSPP_IS_NUMBER IS NOT NULL THEN
                   v_result_rec.RS_TSRLSPP_IS_NUMBER := NULL;
                   v_result_rec.RS_TSRLSPP_ORG_ID := NULL;
                   Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning - Lab Procedure not needed for text type result',NULL,v_activity_seq,NULL);
                END IF;

                IF v_result_rec.RS_DURBASIS_SEQ IS NOT NULL THEN
                   v_result_rec.RS_DURBASIS_SEQ := NULL;
                   Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning - Duration Basis not needed for text type result',NULL,v_activity_seq,NULL);
                END IF;

                IF v_result_rec.RS_TEMPBASIS_SEQ IS NOT NULL THEN
                   v_result_rec.RS_TEMPBASIS_SEQ := NULL;
                   Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning - Temperature Basis not needed for text type result',NULL,v_activity_seq,NULL);
                END IF;

                IF v_result_rec.RS_PRECISION IS NOT NULL THEN
                   v_result_rec.RS_PRECISION := NULL;
                   Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning - Precision not needed for text type result',NULL,v_activity_seq,NULL);
                END IF;

                IF v_result_rec.RS_CONFLEVEL_SEQ IS NOT NULL THEN
                   v_result_rec.RS_CONFLEVEL_SEQ := NULL;
                   Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning - Confidence Level not needed for text type result',NULL,v_activity_seq,NULL);
                END IF;
                v_result_rec.RS_TSRCHGRP_IS_NUMBER := v_tsrchgrp_is_nbr;
                v_result_rec.RS_TSRCHGRP_ORG_ID := v_tsrchgrp_org_id;
                v_result_rec.RS_TSRCHDEF_IS_NUMBER := v_tsrchdef_is_nbr;
                v_result_rec.RS_TSRCHDEF_ORG_ID := v_tsrchdef_org_id;

            END IF;
            /***** end added by dcl to account for text char type ******/

          /* kms - make sure flag to skip result has not been raised */
          --SF - Took out the ability to skip a result. If it's bad, just raise an error.
 --         IF NOT v_skip_result AND v_result IS NOT NULL THEN          
          IF v_result IS NOT NULL THEN
            /* insert result */
            v_success := create_result(v_result_rec);

            /* if insert function failed, write error and prevent activity from being exported */
            IF v_success != 1 THEN
              UPDATE SIM_FIELD_ACTIVITIES
                SET fa_status = 'N'
              WHERE fa_seq = v_activity_seq;

              Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Failed to create result in sim_results, activity will not be exported.',NULL,v_activity_seq,NULL);

            END IF;

            /* Update status of the field activity */
            IF v_status = 'N' THEN
              UPDATE SIM_FIELD_ACTIVITIES
                SET fa_status = 'N'
              WHERE fa_seq = v_activity_seq;
            END IF;
          ELSE
            /* kms - flag to skip result has been raised or there is no result, increment counter and issue warning. */
            v_skipped_results_cnt := v_skipped_results_cnt + 1;
            Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': Warning: Result/Line skipped.',NULL,v_activity_seq,NULL);
          END IF;

          /* wipe out record containing result data */
          v_result_rec := v_result_copy_rec;
          v_acode_defs_rec := v_acode_copy_rec;
          v_trip_rec := v_trip_copy_rec;
          v_visit_rec := v_visit_copy_rec;
        EXCEPTION
          WHEN OTHERS THEN
            v_import_desc := 'Line '||v_line_cnt||': '||SQLERRM||' - writing a result.';
            Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq, v_import_desc,NULL,v_activity_seq,NULL);
        END;

        /* update import lines table with activity and result sequences */
        UPDATE SIM_IMP_LINES SET SIPL_FA_SEQ = v_activity_seq
          WHERE SIPL_SEQ = v_text_line.SIPL_SEQ;

        UPDATE SIM_IMP_LINES SET SIPL_RS_SEQ = v_result_rec.RS_SEQ
          WHERE SIPL_SEQ = v_text_line.SIPL_SEQ;

        /* wipe out record containing field activity data */
        v_fa_rec := v_fa_copy_rec;
      EXCEPTION
        WHEN OTHERS THEN
          v_skipped_results_cnt := v_skipped_results_cnt + 1;
          Sim2_Import_Pkg.write_error(v_import_log_seq, 'Line '||v_line_cnt||': '||SQLERRM||' - processing line at item: '||v_item_name||'.  Line has been skipped.',NULL,v_activity_seq,NULL);
      END;

    END LOOP; /* end of text loop */

    BEGIN
      OPEN c_comp_count(v_import_log_seq);
      FETCH c_comp_count INTO v_complete_cnt;
      CLOSE c_comp_count;

      v_error_cnt := v_line_cnt - v_complete_cnt;

      /* update counts in the import log */
      UPDATE SIM_IMPORT_LOG
         SET SIL_ROWS_IMPORTED = v_line_cnt,
             SIL_ROWS_ERRORS = v_error_cnt,
             SIL_ROWS_READY = v_complete_cnt,
             SIL_LAB_ID = v_first_lab_id
         WHERE SIL_SEQ = v_import_log_seq;

      /* update tsmeisn table for trips */
      /* kms (6/21/01) - should no longer be writing anything to tsrtrip, this is not needed. */
--      IF v_cur_trip_is_nbr IS NOT NULL THEN
--        sim2_export_pkg.update_tsmeisn(p_org_id, 'TSRTRIP ', v_cur_trip_is_nbr);
--      END IF;

      /* update tsmeisn table for projects 
      IF v_cur_proj_is_nbr IS NOT NULL THEN
        Sim2_Export_Pkg.update_tsmeisn(p_org_id, 'TSMPROJ ', v_cur_proj_is_nbr);
      END IF; */

      v_result_cnt := v_line_cnt-v_skipped_results_cnt;
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq,v_line_cnt||' lines processed, '||v_skipped_results_cnt||' results skipped.  '||v_activity_cnt||' activities, '||v_result_cnt||' results loaded.',NULL,v_activity_seq,NULL);
      Sim2_Import_Pkg.WRITE_ERROR(v_import_log_seq ,'Finished at '||TO_CHAR(SYSDATE,'HH24:MI:SS'),NULL,v_activity_seq,NULL);

      COMMIT;
    EXCEPTION
      WHEN OTHERS THEN
        Sim2_Import_Pkg.WRITE_ERROR (v_import_log_seq ,'Error in post line processing: '||SQLERRM,NULL,v_activity_seq,NULL);

    END;
  END;

EXCEPTION

  WHEN e_no_cfg THEN
    RAISE_APPLICATION_ERROR(-20102, 'Configuration invalid or does not exist.');

  WHEN e_bad_delimiter THEN
    RAISE_APPLICATION_ERROR(-20197,'The import file contains the wrong number of columns on line: '||v_line_cnt);

  WHEN NO_DATA_FOUND THEN
    Sim2_Import_Pkg.WRITE_ERROR (V_import_log_seq ,SUBSTR('Line '||v_line_cnt||', Last Item Processed: '||v_item_name||', NO_DATA_FOUND : '||SQLERRM||' - IN Sim2_Import_Result_Pkg.import_results.',1,240),NULL,v_activity_seq,NULL);
    Sim2_Import_Pkg.WRITE_ERROR (V_import_log_seq ,TO_CHAR(v_line_cnt)||' lines processed /'||TO_CHAR(v_activity_cnt)||' activities loaded',NULL,v_activity_seq,NULL);
    Sim2_Import_Pkg.WRITE_ERROR (V_import_log_seq ,' Finished at '||TO_CHAR(SYSDATE,'HH24:MI:SS'),NULL,v_activity_seq,NULL);

  WHEN OTHERS THEN
    Sim2_Import_Pkg.WRITE_ERROR (V_import_log_seq ,SUBSTR('Line '||v_line_cnt||', Last Item Processed: '||v_item_name||', Fatal Error : '||SQLERRM||' - IN Sim2_Import_Result_Pkg.import_results.',1,240),NULL,v_activity_seq,NULL);
    Sim2_Import_Pkg.WRITE_ERROR (V_import_log_seq ,TO_CHAR(v_line_cnt)||' lines processed /'||TO_CHAR(v_activity_cnt)||' activities loaded',NULL,v_activity_seq,NULL);
    Sim2_Import_Pkg.WRITE_ERROR (V_import_log_seq ,' Finished at '||TO_CHAR(SYSDATE,'HH24:MI:SS'),NULL,v_activity_seq,NULL);

END;

/******************************************************************************************************/
END Sim2_Import_Result_Pkg;
/
